#!/usr/bin/env python

import nsysstats

class VulkanMarkerSummary(nsysstats.Report):

    usage = f"""{{SCRIPT}} -- Vulkan Range Summary

    No arguments.

    Output: All time values default to nanoseconds
        Time(%) : Percentage of "Total Time"
        Total Time : The total time used by all instances of this range
        Instances: The number of instances of this range
        Average : The average execution time of this range
        Minimum : The smallest execution time of this range
        Maximum : The largest execution time of this range
        StdDev : The standard deviation of execution time of this range
        Range : The name of the range

    This report provides a summary of Vulkan debug markers on the CPU,
    and their execution times. Note that the "Time(%)" column
    is calculated using a summation of the "Total Time" column, and represents
    that range's percent of the execution time of the ranges listed, and not a
    percentage of the application wall or CPU execution time.
"""

    query = f"""
WITH
    beginEvents AS (
        SELECT
            *
        FROM
            VULKAN_DEBUG_API
        WHERE textId IS NOT NULL
    ),
    endEvents AS (
        SELECT
            *
        FROM
            VULKAN_DEBUG_API
        WHERE textId IS NULL
    ),
    summary AS (
        SELECT
            beginEvents.textId AS textId,
            sum(endEvents.start - beginEvents.end) AS total,
            count(*) AS num,
            avg(endEvents.start - beginEvents.end) AS avg,
            min(endEvents.start - beginEvents.end) AS min,
            max(endEvents.start - beginEvents.end) AS max,
            stdev(endEvents.start - beginEvents.end) AS stddev
        FROM
            beginEvents
        INNER JOIN
            endEvents
            ON beginEvents.correlationId == endEvents.correlationId
        GROUP BY 1
    ),
    totals AS (
        SELECT sum(total) AS total
        FROM summary
    )
SELECT
    round(summary.total * 100.0 / (SELECT total FROM totals), 1) AS "Time(%)",
    summary.total AS "Total Time:dur_ns",
    summary.num AS "Instances",
    round(summary.avg, 1) AS "Average:dur_ns",
    summary.min AS "Minimum:dur_ns",
    summary.max AS "Maximum:dur_ns",
    round(summary.stddev, 1) AS "StdDev:dur_ns",
    ids.value AS "Range"
FROM
    summary
LEFT JOIN
    StringIds AS ids
    ON ids.id == summary.textId
WHERE summary.total > 0
ORDER BY 2 DESC
;
"""

    table_checks = {
        'VULKAN_DEBUG_API':
            "{DBFILE} does not contain Vulkan Debug Extension (Vulkan Debug Util) data"
    }

if __name__ == "__main__":
    VulkanMarkerSummary.Main()
