#ifndef __MFIFO_H
#define __MFIFO_H

#include <linux/stddef.h>
#include <linux/circ_buf.h>
#include <linux/linkage.h>
#include <linux/bitops.h>

#define MFIFO_OK	0
#define MFIFO_BINARY	1
#define MFIFO_U8	8
#define MFIFO_U16	9
#define MFIFO_U32	10

typedef struct _mfifo mfifo_t;

struct mfifo_ops {
	void (*set) (mfifo_t *fifo, void *el);
	void (*get) (mfifo_t *fifo, void *el);

	void (*wrap) (mfifo_t *fifo, int *ending);
	int  (*space) (mfifo_t *fifo);
};

struct _mfifo {
	/* struct circ_buf header; */
	char *buf;
	int head;
	int tail;

	int size;
	int esize;
	int mask;

	struct mfifo_ops ops;
	unsigned long flags;
};

static inline int mfifo_ok (mfifo_t *fifo) {
	return test_bit (MFIFO_OK, &fifo->flags);
}

static inline void mfifo_next (mfifo_t *fifo) {
	fifo->tail++;
	fifo->ops.wrap (fifo, &fifo->tail);
}

static inline void mfifo_put (mfifo_t *fifo, void *el) {
	if (! mfifo_ok (fifo) || ! el)
		return;

	fifo->ops.set (fifo, el);
	fifo->head++;
	fifo->ops.wrap (fifo, &fifo->head);
}

static inline void mfifo_flush (mfifo_t *fifo) {
	fifo->head = fifo->tail = 0;
}

static inline int mfifo_full (mfifo_t *fifo) {
	return 0 == fifo->ops.space (fifo);
}

static inline int mfifo_empty (mfifo_t *fifo) {
	return fifo->head == fifo->tail;
}

static inline void *mfifo_tail (mfifo_t *fifo) {
	return mfifo_ok (fifo) ? fifo->buf + fifo->esize * fifo->tail
			      : NULL;
}

static inline void mfifo_get (mfifo_t *fifo, void *el) {
	if (! mfifo_ok (fifo) || ! el)
		return;

	fifo->ops.get (fifo, el);
	/* fifo->next (fifo);*/
	fifo->tail++;
	fifo->ops.wrap (fifo, &fifo->tail);
}

/* precond: ! mfifo_empty (fifo) */
static inline void *mfifo_lastwr (mfifo_t *fifo) {
	int lastwr = fifo->head - 1;
	fifo->ops.wrap (fifo, &lastwr);
	return  fifo->buf + lastwr * fifo->esize;
}

static inline void *mfifo_head (mfifo_t *fifo) {
	return mfifo_ok (fifo) ? fifo->buf + fifo->head * fifo->esize
			: NULL;
}

void mfifo_init (mfifo_t *fifo, void *_buf, int _size, int _esize); 

#define MFIFO_INIT(fifo, buffer) 				\
		mfifo_init (fifo, buffer, 			\
			   (sizeof buffer / sizeof buffer[0]),	\
			    sizeof buffer[0]);

#define MFIFO_DUMP(mdev, fifo) 								\
	MINFO (#fifo ": %dx%d  %s%s -> (%d, %d)\n", (mdev)->fifo.size, (mdev)->fifo.esize,\
		test_bit (MFIFO_BINARY, &(mdev)->fifo.flags) ? "b | " : "", 		\
		test_bit (MFIFO_U8, &(mdev)->fifo.flags) ? "u8" :			\
		test_bit (MFIFO_U16, &(mdev)->fifo.flags) ? "u16" :			\
		test_bit (MFIFO_U32, &(mdev)->fifo.flags) ? "u32" :			\
		"any",									\
		(mdev)->fifo.head, (mdev)->fifo.tail);

#endif
