"""Apport package hook for the Debian installer.

Copyright (C) 2011 Canonical Ltd.
Authors: Colin Watson <cjwatson@ubuntu.com>,
         Brian Murray <brian@ubuntu.com>
"""  # noqa: D208

# pylint: disable=invalid-name
# pylint: enable=invalid-name

import os

from apport.hookutils import (
    attach_hardware,
    attach_root_command_outputs,
    command_available,
    command_output,
)


def _add_installation_log(report, ident, name):
    if os.path.exists(f"/var/log/installer/{name}"):
        log_file = f"/var/log/installer/{name}"
    elif os.path.exists(f"/var/log/{name}"):
        log_file = f"/var/log/{name}"
    else:
        return

    if os.access(log_file, os.R_OK):
        report[ident] = (log_file,)
    else:
        attach_root_command_outputs(report, {ident: f"cat '{log_file}'"})


def add_info(report):
    """Add Debian installer information."""
    attach_hardware(report)

    report["DiskUsage"] = command_output(["df"])
    report["MemoryUsage"] = command_output(["free"])

    if command_available("dmraid"):
        attach_root_command_outputs(
            report, {"DmraidSets": "dmraid -s", "DmraidDevices": "dmraid -r"}
        )
        if command_available("dmsetup"):
            attach_root_command_outputs(report, {"DeviceMapperTables": "dmsetup table"})

    try:
        with open("/var/log/installer/version", encoding="utf-8") as installer_version:
            for line in installer_version:
                if line.startswith("ubiquity "):
                    # File these reports on the ubiquity package instead
                    report["SourcePackage"] = "ubiquity"
                    break
    except IOError:
        pass

    _add_installation_log(report, "DIPartman", "partman")
    _add_installation_log(report, "DISyslog", "syslog")


def main():  # pylint: disable=missing-function-docstring
    report = {}
    add_info(report)
    for key, value in report.items():
        if isinstance(value, str):
            print("%s: %s" % (key, value.split("\n", 1)[0]))
        else:
            print(f"{key}: {type(value)}")


if __name__ == "__main__":
    main()
