# Licensed under the Apache License: http://www.apache.org/licenses/LICENSE-2.0
# For details: https://bitbucket.org/ned/coveragepy/src/default/NOTICE.txt

"""Tests of miscellaneous stuff."""

import sys

import pytest

import coverage
from coverage.version import _make_url, _make_version
from coverage.misc import contract, dummy_decorator_with_args, file_be_gone, Hasher, one_of

from tests.coveragetest import CoverageTest


class HasherTest(CoverageTest):
    """Test our wrapper of md5 hashing."""

    run_in_temp_dir = False

    def test_string_hashing(self):
        h1 = Hasher()
        h1.update("Hello, world!")
        h2 = Hasher()
        h2.update("Goodbye!")
        h3 = Hasher()
        h3.update("Hello, world!")
        self.assertNotEqual(h1.hexdigest(), h2.hexdigest())
        self.assertEqual(h1.hexdigest(), h3.hexdigest())

    def test_bytes_hashing(self):
        h1 = Hasher()
        h1.update(b"Hello, world!")
        h2 = Hasher()
        h2.update(b"Goodbye!")
        self.assertNotEqual(h1.hexdigest(), h2.hexdigest())

    def test_unicode_hashing(self):
        h1 = Hasher()
        h1.update(u"Hello, world! \N{SNOWMAN}")
        h2 = Hasher()
        h2.update(u"Goodbye!")
        self.assertNotEqual(h1.hexdigest(), h2.hexdigest())

    def test_dict_hashing(self):
        h1 = Hasher()
        h1.update({'a': 17, 'b': 23})
        h2 = Hasher()
        h2.update({'b': 23, 'a': 17})
        self.assertEqual(h1.hexdigest(), h2.hexdigest())


class RemoveFileTest(CoverageTest):
    """Tests of misc.file_be_gone."""

    def test_remove_nonexistent_file(self):
        # It's OK to try to remove a file that doesn't exist.
        file_be_gone("not_here.txt")

    def test_remove_actual_file(self):
        # It really does remove a file that does exist.
        self.make_file("here.txt", "We are here, we are here, we are here!")
        file_be_gone("here.txt")
        self.assert_doesnt_exist("here.txt")

    def test_actual_errors(self):
        # Errors can still happen.
        # ". is a directory" on Unix, or "Access denied" on Windows
        with self.assertRaises(OSError):
            file_be_gone(".")


class ContractTest(CoverageTest):
    """Tests of our contract decorators."""

    run_in_temp_dir = False

    def test_bytes(self):
        @contract(text='bytes|None')
        def need_bytes(text=None):                      # pylint: disable=missing-docstring
            return text

        assert need_bytes(b"Hey") == b"Hey"
        assert need_bytes() is None
        with pytest.raises(Exception):
            need_bytes(u"Oops")

    def test_unicode(self):
        @contract(text='unicode|None')
        def need_unicode(text=None):                    # pylint: disable=missing-docstring
            return text

        assert need_unicode(u"Hey") == u"Hey"
        assert need_unicode() is None
        with pytest.raises(Exception):
            need_unicode(b"Oops")

    def test_one_of(self):
        @one_of("a, b, c")
        def give_me_one(a=None, b=None, c=None):        # pylint: disable=missing-docstring
            return (a, b, c)

        assert give_me_one(a=17) == (17, None, None)
        assert give_me_one(b=set()) == (None, set(), None)
        assert give_me_one(c=17) == (None, None, 17)
        with pytest.raises(AssertionError):
            give_me_one(a=17, b=set())
        with pytest.raises(AssertionError):
            give_me_one()

    def test_dummy_decorator_with_args(self):
        @dummy_decorator_with_args("anything", this=17, that="is fine")
        def undecorated(a=None, b=None):                # pylint: disable=missing-docstring
            return (a, b)

        assert undecorated() == (None, None)
        assert undecorated(17) == (17, None)
        assert undecorated(b=23) == (None, 23)
        assert undecorated(b=42, a=3) == (3, 42)


class VersionTest(CoverageTest):
    """Tests of version.py"""

    run_in_temp_dir = False

    def test_version_info(self):
        # Make sure we didn't screw up the version_info tuple.
        self.assertIsInstance(coverage.version_info, tuple)
        self.assertEqual([type(d) for d in coverage.version_info], [int, int, int, str, int])
        self.assertIn(coverage.version_info[3], ['alpha', 'beta', 'candidate', 'final'])

    def test_make_version(self):
        self.assertEqual(_make_version(4, 0, 0, 'alpha', 0), "4.0a0")
        self.assertEqual(_make_version(4, 0, 0, 'alpha', 1), "4.0a1")
        self.assertEqual(_make_version(4, 0, 0, 'final', 0), "4.0")
        self.assertEqual(_make_version(4, 1, 2, 'beta', 3), "4.1.2b3")
        self.assertEqual(_make_version(4, 1, 2, 'final', 0), "4.1.2")
        self.assertEqual(_make_version(5, 10, 2, 'candidate', 7), "5.10.2rc7")

    def test_make_url(self):
        self.assertEqual(
            _make_url(4, 0, 0, 'final', 0),
            "https://coverage.readthedocs.io"
        )
        self.assertEqual(
            _make_url(4, 1, 2, 'beta', 3),
            "https://coverage.readthedocs.io/en/coverage-4.1.2b3"
        )


class SetupPyTest(CoverageTest):
    """Tests of setup.py"""

    run_in_temp_dir = False

    def setUp(self):
        super(SetupPyTest, self).setUp()
        # Force the most restrictive interpretation.
        self.set_environ('LC_ALL', 'C')

    def test_metadata(self):
        status, output = self.run_command_status(
            "python setup.py --description --version --url --author"
            )
        self.assertEqual(status, 0)
        out = output.splitlines()
        self.assertIn("measurement", out[0])
        self.assertEqual(out[1], coverage.__version__)
        self.assertEqual(out[2], coverage.__url__)
        self.assertIn("Ned Batchelder", out[3])

    def test_more_metadata(self):
        # Let's be sure we pick up our own setup.py
        # CoverageTest restores the original sys.path for us.
        sys.path.insert(0, '')
        from setup import setup_args

        classifiers = setup_args['classifiers']
        self.assertGreater(len(classifiers), 7)
        self.assert_starts_with(classifiers[-1], "Development Status ::")

        long_description = setup_args['long_description'].splitlines()
        self.assertGreater(len(long_description), 7)
        self.assertNotEqual(long_description[0].strip(), "")
        self.assertNotEqual(long_description[-1].strip(), "")
