# vim: tabstop=4 shiftwidth=4 softtabstop=4

# Copyright (c) 2011 X.commerce, a business unit of eBay Inc.
# Copyright 2010 United States Government as represented by the
# Administrator of the National Aeronautics and Space Administration.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Implements vlans, bridges, and iptables rules using linux utilities."""

import calendar
import inspect
import netaddr
import os
import re

from oslo.config import cfg

from nova import db
from nova import exception
from nova.openstack.common import excutils
from nova.openstack.common import fileutils
from nova.openstack.common import importutils
from nova.openstack.common import jsonutils
from nova.openstack.common import lockutils
from nova.openstack.common import log as logging
from nova.openstack.common import timeutils
from nova import paths
from nova import utils

LOG = logging.getLogger(__name__)


linux_net_opts = [
    cfg.MultiStrOpt('dhcpbridge_flagfile',
                    default=['/etc/nova/nova-dhcpbridge.conf'],
                    help='location of flagfiles for dhcpbridge'),
    cfg.StrOpt('networks_path',
               default=paths.state_path_def('networks'),
               help='Location to keep network config files'),
    cfg.StrOpt('public_interface',
               default='eth0',
               help='Interface for public IP addresses'),
    cfg.StrOpt('network_device_mtu',
               default=None,
               help='MTU setting for vlan'),
    cfg.StrOpt('dhcpbridge',
               default=paths.bindir_def('nova-dhcpbridge'),
               help='location of nova-dhcpbridge'),
    cfg.StrOpt('routing_source_ip',
               default='$my_ip',
               help='Public IP of network host'),
    cfg.IntOpt('dhcp_lease_time',
               default=120,
               help='Lifetime of a DHCP lease in seconds'),
    cfg.MultiStrOpt('dns_server',
                    default=[],
                    help='if set, uses specific dns server for dnsmasq. Can'
                         'be specified multiple times.'),
    cfg.BoolOpt('use_network_dns_servers',
                default=False,
                help='if set, uses the dns1 and dns2 from the network ref.'
                     'as dns servers.'),
    cfg.ListOpt('dmz_cidr',
               default=[],
               help='A list of dmz range that should be accepted'),
    cfg.MultiStrOpt('force_snat_range',
               default=[],
               help='Traffic to this range will always be snatted to the '
                    'fallback ip, even if it would normally be bridged out '
                    'of the node. Can be specified multiple times.'),
    cfg.StrOpt('dnsmasq_config_file',
               default='',
               help='Override the default dnsmasq settings with this file'),
    cfg.StrOpt('linuxnet_interface_driver',
               default='nova.network.linux_net.LinuxBridgeInterfaceDriver',
               help='Driver used to create ethernet devices.'),
    cfg.StrOpt('linuxnet_ovs_integration_bridge',
               default='br-int',
               help='Name of Open vSwitch bridge used with linuxnet'),
    cfg.BoolOpt('send_arp_for_ha',
                default=False,
                help='send gratuitous ARPs for HA setup'),
    cfg.IntOpt('send_arp_for_ha_count',
               default=3,
               help='send this many gratuitous ARPs for HA setup'),
    cfg.BoolOpt('use_single_default_gateway',
                default=False,
                help='Use single default gateway. Only first nic of vm will '
                     'get default gateway from dhcp server'),
    cfg.MultiStrOpt('forward_bridge_interface',
                    default=['all'],
                    help='An interface that bridges can forward to. If this '
                         'is set to all then all traffic will be forwarded. '
                         'Can be specified multiple times.'),
    cfg.StrOpt('metadata_host',
               default='$my_ip',
               help='the ip for the metadata api server'),
    cfg.IntOpt('metadata_port',
               default=8775,
               help='the port for the metadata api port'),
    cfg.StrOpt('iptables_top_regex',
               default='',
               help='Regular expression to match iptables rule that should'
                    'always be on the top.'),
    cfg.StrOpt('iptables_bottom_regex',
               default='',
               help='Regular expression to match iptables rule that should'
                    'always be on the bottom.'),
    ]

CONF = cfg.CONF
CONF.register_opts(linux_net_opts)
CONF.import_opt('host', 'nova.netconf')
CONF.import_opt('use_ipv6', 'nova.netconf')
CONF.import_opt('my_ip', 'nova.netconf')


# NOTE(vish): Iptables supports chain names of up to 28 characters,  and we
#             add up to 12 characters to binary_name which is used as a prefix,
#             so we limit it to 16 characters.
#             (max_chain_name_length - len('-POSTROUTING') == 16)
def get_binary_name():
    """Grab the name of the binary we're running in."""
    return os.path.basename(inspect.stack()[-1][1])[:16]

binary_name = get_binary_name()


class IptablesRule(object):
    """An iptables rule.

    You shouldn't need to use this class directly, it's only used by
    IptablesManager.

    """

    def __init__(self, chain, rule, wrap=True, top=False):
        self.chain = chain
        self.rule = rule
        self.wrap = wrap
        self.top = top

    def __eq__(self, other):
        return ((self.chain == other.chain) and
                (self.rule == other.rule) and
                (self.top == other.top) and
                (self.wrap == other.wrap))

    def __ne__(self, other):
        return not self == other

    def __str__(self):
        if self.wrap:
            chain = '%s-%s' % (binary_name, self.chain)
        else:
            chain = self.chain
        # new rules should have a zero [packet: byte] count
        return '[0:0] -A %s %s' % (chain, self.rule)


class IptablesTable(object):
    """An iptables table."""

    def __init__(self):
        self.rules = []
        self.remove_rules = []
        self.chains = set()
        self.unwrapped_chains = set()
        self.remove_chains = set()
        self.dirty = True

    def add_chain(self, name, wrap=True):
        """Adds a named chain to the table.

        The chain name is wrapped to be unique for the component creating
        it, so different components of Nova can safely create identically
        named chains without interfering with one another.

        At the moment, its wrapped name is <binary name>-<chain name>,
        so if nova-compute creates a chain named 'OUTPUT', it'll actually
        end up named 'nova-compute-OUTPUT'.

        """
        if wrap:
            self.chains.add(name)
        else:
            self.unwrapped_chains.add(name)
        self.dirty = True

    def remove_chain(self, name, wrap=True):
        """Remove named chain.

        This removal "cascades". All rule in the chain are removed, as are
        all rules in other chains that jump to it.

        If the chain is not found, this is merely logged.

        """
        if wrap:
            chain_set = self.chains
        else:
            chain_set = self.unwrapped_chains

        if name not in chain_set:
            LOG.warn(_('Attempted to remove chain %s which does not exist'),
                     name)
            return
        self.dirty = True

        # non-wrapped chains and rules need to be dealt with specially,
        # so we keep a list of them to be iterated over in apply()
        if not wrap:
            self.remove_chains.add(name)
        chain_set.remove(name)
        if not wrap:
            self.remove_rules += filter(lambda r: r.chain == name, self.rules)
        self.rules = filter(lambda r: r.chain != name, self.rules)

        if wrap:
            jump_snippet = '-j %s-%s' % (binary_name, name)
        else:
            jump_snippet = '-j %s' % (name,)

        if not wrap:
            self.remove_rules += filter(lambda r: jump_snippet in r.rule,
                                        self.rules)
        self.rules = filter(lambda r: jump_snippet not in r.rule, self.rules)

    def add_rule(self, chain, rule, wrap=True, top=False):
        """Add a rule to the table.

        This is just like what you'd feed to iptables, just without
        the '-A <chain name>' bit at the start.

        However, if you need to jump to one of your wrapped chains,
        prepend its name with a '$' which will ensure the wrapping
        is applied correctly.

        """
        if wrap and chain not in self.chains:
            raise ValueError(_('Unknown chain: %r') % chain)

        if '$' in rule:
            rule = ' '.join(map(self._wrap_target_chain, rule.split(' ')))

        rule_obj = IptablesRule(chain, rule, wrap, top)
        if rule_obj in self.rules:
            LOG.debug(_("Skipping duplicate iptables rule addition"))
        else:
            self.rules.append(IptablesRule(chain, rule, wrap, top))
            self.dirty = True

    def _wrap_target_chain(self, s):
        if s.startswith('$'):
            return '%s-%s' % (binary_name, s[1:])
        return s

    def remove_rule(self, chain, rule, wrap=True, top=False):
        """Remove a rule from a chain.

        Note: The rule must be exactly identical to the one that was added.
        You cannot switch arguments around like you can with the iptables
        CLI tool.

        """
        try:
            self.rules.remove(IptablesRule(chain, rule, wrap, top))
            if not wrap:
                self.remove_rules.append(IptablesRule(chain, rule, wrap, top))
            self.dirty = True
        except ValueError:
            LOG.warn(_('Tried to remove rule that was not there:'
                       ' %(chain)r %(rule)r %(wrap)r %(top)r'),
                     {'chain': chain, 'rule': rule,
                      'top': top, 'wrap': wrap})

    def remove_rules_regex(self, regex):
        """Remove all rules matching regex."""
        if isinstance(regex, basestring):
            regex = re.compile(regex)
        num_rules = len(self.rules)
        self.rules = filter(lambda r: not regex.match(str(r)), self.rules)
        removed = num_rules - len(self.rules)
        if removed > 0:
            self.dirty = True
        return removed

    def empty_chain(self, chain, wrap=True):
        """Remove all rules from a chain."""
        chained_rules = [rule for rule in self.rules
                              if rule.chain == chain and rule.wrap == wrap]
        if chained_rules:
            self.dirty = True
        for rule in chained_rules:
            self.rules.remove(rule)


class IptablesManager(object):
    """Wrapper for iptables.

    See IptablesTable for some usage docs

    A number of chains are set up to begin with.

    First, nova-filter-top. It's added at the top of FORWARD and OUTPUT. Its
    name is not wrapped, so it's shared between the various nova workers. It's
    intended for rules that need to live at the top of the FORWARD and OUTPUT
    chains. It's in both the ipv4 and ipv6 set of tables.

    For ipv4 and ipv6, the built-in INPUT, OUTPUT, and FORWARD filter chains
    are wrapped, meaning that the "real" INPUT chain has a rule that jumps to
    the wrapped INPUT chain, etc. Additionally, there's a wrapped chain named
    "local" which is jumped to from nova-filter-top.

    For ipv4, the built-in PREROUTING, OUTPUT, and POSTROUTING nat chains are
    wrapped in the same was as the built-in filter chains. Additionally,
    there's a snat chain that is applied after the POSTROUTING chain.

    """

    def __init__(self, execute=None):
        if not execute:
            self.execute = _execute
        else:
            self.execute = execute

        self.ipv4 = {'filter': IptablesTable(),
                     'nat': IptablesTable(),
                     'mangle': IptablesTable()}
        self.ipv6 = {'filter': IptablesTable()}

        self.iptables_apply_deferred = False

        # Add a nova-filter-top chain. It's intended to be shared
        # among the various nova components. It sits at the very top
        # of FORWARD and OUTPUT.
        for tables in [self.ipv4, self.ipv6]:
            tables['filter'].add_chain('nova-filter-top', wrap=False)
            tables['filter'].add_rule('FORWARD', '-j nova-filter-top',
                                      wrap=False, top=True)
            tables['filter'].add_rule('OUTPUT', '-j nova-filter-top',
                                      wrap=False, top=True)

            tables['filter'].add_chain('local')
            tables['filter'].add_rule('nova-filter-top', '-j $local',
                                      wrap=False)

        # Wrap the built-in chains
        builtin_chains = {4: {'filter': ['INPUT', 'OUTPUT', 'FORWARD'],
                              'nat': ['PREROUTING', 'OUTPUT', 'POSTROUTING'],
                              'mangle': ['POSTROUTING']},
                          6: {'filter': ['INPUT', 'OUTPUT', 'FORWARD']}}

        for ip_version in builtin_chains:
            if ip_version == 4:
                tables = self.ipv4
            elif ip_version == 6:
                tables = self.ipv6

            for table, chains in builtin_chains[ip_version].iteritems():
                for chain in chains:
                    tables[table].add_chain(chain)
                    tables[table].add_rule(chain, '-j $%s' % (chain,),
                                           wrap=False)

        # Add a nova-postrouting-bottom chain. It's intended to be shared
        # among the various nova components. We set it as the last chain
        # of POSTROUTING chain.
        self.ipv4['nat'].add_chain('nova-postrouting-bottom', wrap=False)
        self.ipv4['nat'].add_rule('POSTROUTING', '-j nova-postrouting-bottom',
                                  wrap=False)

        # We add a snat chain to the shared nova-postrouting-bottom chain
        # so that it's applied last.
        self.ipv4['nat'].add_chain('snat')
        self.ipv4['nat'].add_rule('nova-postrouting-bottom', '-j $snat',
                                  wrap=False)

        # And then we add a float-snat chain and jump to first thing in
        # the snat chain.
        self.ipv4['nat'].add_chain('float-snat')
        self.ipv4['nat'].add_rule('snat', '-j $float-snat')

    def defer_apply_on(self):
        self.iptables_apply_deferred = True

    def defer_apply_off(self):
        self.iptables_apply_deferred = False
        self.apply()

    def dirty(self):
        for table in self.ipv4.itervalues():
            if table.dirty:
                return True
        if CONF.use_ipv6:
            for table in self.ipv6.itervalues():
                if table.dirty:
                    return True
        return False

    def apply(self):
        if self.iptables_apply_deferred:
            return
        if self.dirty():
            self._apply()
        else:
            LOG.debug(_("Skipping apply due to lack of new rules"))

    @lockutils.synchronized('iptables', 'nova-', external=True)
    def _apply(self):
        """Apply the current in-memory set of iptables rules.

        This will blow away any rules left over from previous runs of the
        same component of Nova, and replace them with our current set of
        rules. This happens atomically, thanks to iptables-restore.

        """
        s = [('iptables', self.ipv4)]
        if CONF.use_ipv6:
            s += [('ip6tables', self.ipv6)]

        for cmd, tables in s:
            all_tables, _err = self.execute('%s-save' % (cmd,), '-c',
                                                run_as_root=True,
                                                attempts=5)
            all_lines = all_tables.split('\n')
            for table_name, table in tables.iteritems():
                start, end = self._find_table(all_lines, table_name)
                all_lines[start:end] = self._modify_rules(
                        all_lines[start:end], table, table_name)
                table.dirty = False
            self.execute('%s-restore' % (cmd,), '-c', run_as_root=True,
                         process_input='\n'.join(all_lines),
                         attempts=5)
        LOG.debug(_("IPTablesManager.apply completed with success"))

    def _find_table(self, lines, table_name):
        if len(lines) < 3:
            # length only <2 when fake iptables
            return (0, 0)
        try:
            start = lines.index('*%s' % table_name) - 1
        except ValueError:
            # Couldn't find table_name
            return (0, 0)
        end = lines[start:].index('COMMIT') + start + 2
        return (start, end)

    def _modify_rules(self, current_lines, table, table_name):
        unwrapped_chains = table.unwrapped_chains
        chains = table.chains
        remove_chains = table.remove_chains
        rules = table.rules
        remove_rules = table.remove_rules

        if not current_lines:
            fake_table = ['#Generated by nova',
                          '*' + table_name, 'COMMIT',
                          '#Completed by nova']
            current_lines = fake_table

        # Remove any trace of our rules
        new_filter = filter(lambda line: binary_name not in line,
                            current_lines)

        top_rules = []
        bottom_rules = []

        if CONF.iptables_top_regex:
            regex = re.compile(CONF.iptables_top_regex)
            temp_filter = filter(lambda line: regex.search(line), new_filter)
            for rule_str in temp_filter:
                new_filter = filter(lambda s: s.strip() != rule_str.strip(),
                                    new_filter)
            top_rules = temp_filter

        if CONF.iptables_bottom_regex:
            regex = re.compile(CONF.iptables_bottom_regex)
            temp_filter = filter(lambda line: regex.search(line), new_filter)
            for rule_str in temp_filter:
                new_filter = filter(lambda s: s.strip() != rule_str.strip(),
                    new_filter)
            bottom_rules = temp_filter

        seen_chains = False
        rules_index = 0
        for rules_index, rule in enumerate(new_filter):
            if not seen_chains:
                if rule.startswith(':'):
                    seen_chains = True
            else:
                if not rule.startswith(':'):
                    break

        if not seen_chains:
            rules_index = 2

        our_rules = top_rules
        bot_rules = []
        for rule in rules:
            rule_str = str(rule)
            if rule.top:
                # rule.top == True means we want this rule to be at the top.
                # Further down, we weed out duplicates from the bottom of the
                # list, so here we remove the dupes ahead of time.

                # We don't want to remove an entry if it has non-zero
                # [packet:byte] counts and replace it with [0:0], so let's
                # go look for a duplicate, and over-ride our table rule if
                # found.

                # ignore [packet:byte] counts at beginning of line
                if rule_str.startswith('['):
                    rule_str = rule_str.split(']', 1)[1]
                dup_filter = filter(lambda s: rule_str.strip() in s.strip(),
                                    new_filter)

                new_filter = filter(lambda s:
                                    rule_str.strip() not in s.strip(),
                                    new_filter)
                # if no duplicates, use original rule
                if dup_filter:
                    # grab the last entry, if there is one
                    dup = dup_filter[-1]
                    rule_str = str(dup)
                else:
                    rule_str = str(rule)
                rule_str.strip()

                our_rules += [rule_str]
            else:
                bot_rules += [rule_str]

        our_rules += bot_rules

        new_filter[rules_index:rules_index] = our_rules

        new_filter[rules_index:rules_index] = [':%s - [0:0]' % (name,)
                                               for name in unwrapped_chains]
        new_filter[rules_index:rules_index] = [':%s-%s - [0:0]' %
                                               (binary_name, name,)
                                               for name in chains]

        commit_index = new_filter.index('COMMIT')
        new_filter[commit_index:commit_index] = bottom_rules
        seen_lines = set()

        def _weed_out_duplicates(line):
            # ignore [packet:byte] counts at beginning of lines
            if line.startswith('['):
                line = line.split(']', 1)[1]
            line = line.strip()
            if line in seen_lines:
                return False
            else:
                seen_lines.add(line)
                return True

        def _weed_out_removes(line):
            # We need to find exact matches here
            if line.startswith(':'):
                # it's a chain, for example, ":nova-billing - [0:0]"
                # strip off everything except the chain name
                line = line.split(':')[1]
                line = line.split('- [')[0]
                line = line.strip()
                for chain in remove_chains:
                    if chain == line:
                        remove_chains.remove(chain)
                        return False
            elif line.startswith('['):
                # it's a rule
                # ignore [packet:byte] counts at beginning of lines
                line = line.split(']', 1)[1]
                line = line.strip()
                for rule in remove_rules:
                    # ignore [packet:byte] counts at beginning of rules
                    rule_str = str(rule)
                    rule_str = rule_str.split(' ', 1)[1]
                    rule_str = rule_str.strip()
                    if rule_str == line:
                        remove_rules.remove(rule)
                        return False

            # Leave it alone
            return True

        # We filter duplicates, letting the *last* occurrence take
        # precendence.  We also filter out anything in the "remove"
        # lists.
        new_filter.reverse()
        new_filter = filter(_weed_out_duplicates, new_filter)
        new_filter = filter(_weed_out_removes, new_filter)
        new_filter.reverse()

        # flush lists, just in case we didn't find something
        remove_chains.clear()
        for rule in remove_rules:
            remove_rules.remove(rule)

        return new_filter


# NOTE(jkoelker) This is just a nice little stub point since mocking
#                builtins with mox is a nightmare
def write_to_file(file, data, mode='w'):
    with open(file, mode) as f:
        f.write(data)


def metadata_forward():
    """Create forwarding rule for metadata."""
    if CONF.metadata_host != '127.0.0.1':
        iptables_manager.ipv4['nat'].add_rule('PREROUTING',
                                          '-s 0.0.0.0/0 -d 169.254.169.254/32 '
                                          '-p tcp -m tcp --dport 80 -j DNAT '
                                          '--to-destination %s:%s' %
                                          (CONF.metadata_host,
                                           CONF.metadata_port))
    else:
        iptables_manager.ipv4['nat'].add_rule('PREROUTING',
                                          '-s 0.0.0.0/0 -d 169.254.169.254/32 '
                                          '-p tcp -m tcp --dport 80 '
                                          '-j REDIRECT --to-ports %s' %
                                           CONF.metadata_port)
    iptables_manager.apply()


def metadata_accept():
    """Create the filter accept rule for metadata."""
    iptables_manager.ipv4['filter'].add_rule('INPUT',
                                             '-s 0.0.0.0/0 -d %s '
                                             '-p tcp -m tcp --dport %s '
                                             '-j ACCEPT' %
                                             (CONF.metadata_host,
                                              CONF.metadata_port))
    iptables_manager.apply()


def add_snat_rule(ip_range):
    if CONF.routing_source_ip:
        for dest_range in CONF.force_snat_range or ['0.0.0.0/0']:
            rule = ('-s %s -d %s -j SNAT --to-source %s'
                    % (ip_range, dest_range, CONF.routing_source_ip))
            if CONF.public_interface:
                rule += ' -o %s' % CONF.public_interface
            iptables_manager.ipv4['nat'].add_rule('snat', rule)
        iptables_manager.apply()


def init_host(ip_range=None):
    """Basic networking setup goes here."""
    # NOTE(devcamcar): Cloud public SNAT entries and the default
    # SNAT rule for outbound traffic.
    if not ip_range:
        ip_range = CONF.fixed_range

    add_snat_rule(ip_range)

    rules = []
    for snat_range in CONF.force_snat_range:
        rules.append('PREROUTING -p ipv4 --ip-src %s --ip-dst %s '
                     '-j redirect --redirect-target ACCEPT' %
                     (ip_range, snat_range))
    if rules:
        ensure_ebtables_rules(rules, 'nat')

    iptables_manager.ipv4['nat'].add_rule('POSTROUTING',
                                          '-s %s -d %s/32 -j ACCEPT' %
                                          (ip_range, CONF.metadata_host))

    for dmz in CONF.dmz_cidr:
        iptables_manager.ipv4['nat'].add_rule('POSTROUTING',
                                              '-s %s -d %s -j ACCEPT' %
                                              (ip_range, dmz))

    iptables_manager.ipv4['nat'].add_rule('POSTROUTING',
                                          '-s %(range)s -d %(range)s '
                                          '-m conntrack ! --ctstate DNAT '
                                          '-j ACCEPT' %
                                          {'range': ip_range})
    iptables_manager.apply()


def send_arp_for_ip(ip, device, count):
    out, err = _execute('arping', '-U', ip,
                        '-A', '-I', device,
                        '-c', str(count),
                        run_as_root=True, check_exit_code=False)

    if err:
        LOG.debug(_('arping error for ip %s'), ip)


def bind_floating_ip(floating_ip, device):
    """Bind ip to public interface."""
    _execute('ip', 'addr', 'add', str(floating_ip) + '/32',
             'dev', device,
             run_as_root=True, check_exit_code=[0, 2, 254])

    if CONF.send_arp_for_ha and CONF.send_arp_for_ha_count > 0:
        send_arp_for_ip(floating_ip, device, CONF.send_arp_for_ha_count)


def unbind_floating_ip(floating_ip, device):
    """Unbind a public ip from public interface."""
    _execute('ip', 'addr', 'del', str(floating_ip) + '/32',
             'dev', device,
             run_as_root=True, check_exit_code=[0, 2, 254])


def ensure_metadata_ip():
    """Sets up local metadata ip."""
    _execute('ip', 'addr', 'add', '169.254.169.254/32',
             'scope', 'link', 'dev', 'lo',
             run_as_root=True, check_exit_code=[0, 2, 254])


def ensure_vpn_forward(public_ip, port, private_ip):
    """Sets up forwarding rules for vlan."""
    iptables_manager.ipv4['filter'].add_rule('FORWARD',
                                             '-d %s -p udp '
                                             '--dport 1194 '
                                             '-j ACCEPT' % private_ip)
    iptables_manager.ipv4['nat'].add_rule('PREROUTING',
                                          '-d %s -p udp '
                                          '--dport %s -j DNAT --to %s:1194' %
                                          (public_ip, port, private_ip))
    iptables_manager.ipv4['nat'].add_rule('OUTPUT',
                                          '-d %s -p udp '
                                          '--dport %s -j DNAT --to %s:1194' %
                                          (public_ip, port, private_ip))
    iptables_manager.apply()


def ensure_floating_forward(floating_ip, fixed_ip, device, network):
    """Ensure floating ip forwarding rule."""
    # NOTE(vish): Make sure we never have duplicate rules for the same ip
    regex = '.*\s+%s(/32|\s+|$)' % floating_ip
    num_rules = iptables_manager.ipv4['nat'].remove_rules_regex(regex)
    if num_rules:
        msg = _('Removed %(num)d duplicate rules for floating ip %(float)s')
        LOG.warn(msg % {'num': num_rules, 'float': floating_ip})
    for chain, rule in floating_forward_rules(floating_ip, fixed_ip, device):
        iptables_manager.ipv4['nat'].add_rule(chain, rule)
    iptables_manager.apply()
    if device != network['bridge']:
        ensure_ebtables_rules(*floating_ebtables_rules(fixed_ip, network))


def remove_floating_forward(floating_ip, fixed_ip, device, network):
    """Remove forwarding for floating ip."""
    for chain, rule in floating_forward_rules(floating_ip, fixed_ip, device):
        iptables_manager.ipv4['nat'].remove_rule(chain, rule)
    iptables_manager.apply()
    if device != network['bridge']:
        remove_ebtables_rules(*floating_ebtables_rules(fixed_ip, network))


def floating_ebtables_rules(fixed_ip, network):
    """Makes sure only in-network traffic is bridged."""
    return (['PREROUTING --logical-in %s -p ipv4 --ip-src %s '
            '! --ip-dst %s -j redirect --redirect-target ACCEPT' %
            (network['bridge'], fixed_ip, network['cidr'])], 'nat')


def floating_forward_rules(floating_ip, fixed_ip, device):
    rules = []
    rule = '-s %s -j SNAT --to %s' % (fixed_ip, floating_ip)
    if device:
        rules.append(('float-snat', rule + ' -d %s' % fixed_ip))
        rules.append(('float-snat', rule + ' -o %s' % device))
    else:
        rules.append(('float-snat', rule))
    rules.append(
            ('PREROUTING', '-d %s -j DNAT --to %s' % (floating_ip, fixed_ip)))
    rules.append(
            ('OUTPUT', '-d %s -j DNAT --to %s' % (floating_ip, fixed_ip)))
    rules.append(('POSTROUTING', '-s %s -m conntrack --ctstate DNAT -j SNAT '
                  '--to-source %s' %
                  (fixed_ip, floating_ip)))
    return rules


def initialize_gateway_device(dev, network_ref):
    if not network_ref:
        return

    _execute('sysctl', '-w', 'net.ipv4.ip_forward=1', run_as_root=True)

    # NOTE(vish): The ip for dnsmasq has to be the first address on the
    #             bridge for it to respond to reqests properly
    full_ip = '%s/%s' % (network_ref['dhcp_server'],
                         network_ref['cidr'].rpartition('/')[2])
    new_ip_params = [[full_ip, 'brd', network_ref['broadcast']]]
    old_ip_params = []
    out, err = _execute('ip', 'addr', 'show', 'dev', dev,
                        'scope', 'global', run_as_root=True)
    for line in out.split('\n'):
        fields = line.split()
        if fields and fields[0] == 'inet':
            ip_params = fields[1:-1]
            old_ip_params.append(ip_params)
            if ip_params[0] != full_ip:
                new_ip_params.append(ip_params)
    if not old_ip_params or old_ip_params[0][0] != full_ip:
        old_routes = []
        result = _execute('ip', 'route', 'show', 'dev', dev,
                          run_as_root=True)
        if result:
            out, err = result
            for line in out.split('\n'):
                fields = line.split()
                if fields and 'via' in fields:
                    old_routes.append(fields)
                    _execute('ip', 'route', 'del', fields[0],
                             'dev', dev, run_as_root=True)
        for ip_params in old_ip_params:
            _execute(*_ip_bridge_cmd('del', ip_params, dev),
                     run_as_root=True, check_exit_code=[0, 2, 254])
        for ip_params in new_ip_params:
            _execute(*_ip_bridge_cmd('add', ip_params, dev),
                     run_as_root=True, check_exit_code=[0, 2, 254])

        for fields in old_routes:
            _execute('ip', 'route', 'add', *fields,
                     run_as_root=True)
        if CONF.send_arp_for_ha and CONF.send_arp_for_ha_count > 0:
            send_arp_for_ip(network_ref['dhcp_server'], dev,
                            CONF.send_arp_for_ha_count)
    if(CONF.use_ipv6):
        _execute('ip', '-f', 'inet6', 'addr',
                 'change', network_ref['cidr_v6'],
                 'dev', dev, run_as_root=True)


def get_dhcp_leases(context, network_ref):
    """Return a network's hosts config in dnsmasq leasefile format."""
    hosts = []
    host = None
    if network_ref['multi_host']:
        host = CONF.host
    for data in db.network_get_associated_fixed_ips(context,
                                                    network_ref['id'],
                                                    host=host):
        # NOTE(cfb): Don't return a lease entry if the IP isn't
        #            already leased
        if data['allocated'] and data['leased']:
            hosts.append(_host_lease(data))

    return '\n'.join(hosts)


def get_dhcp_hosts(context, network_ref):
    """Get network's hosts config in dhcp-host format."""
    hosts = []
    host = None
    if network_ref['multi_host']:
        host = CONF.host
    macs = set()
    for data in db.network_get_associated_fixed_ips(context,
                                                    network_ref['id'],
                                                    host=host):
        if data['vif_address'] not in macs:
            hosts.append(_host_dhcp(data))
            macs.add(data['vif_address'])
    return '\n'.join(hosts)


def get_dns_hosts(context, network_ref):
    """Get network's DNS hosts in hosts format."""
    hosts = []
    for data in db.network_get_associated_fixed_ips(context,
                                                    network_ref['id']):
        hosts.append(_host_dns(data))
    return '\n'.join(hosts)


def _add_dnsmasq_accept_rules(dev):
    """Allow DHCP and DNS traffic through to dnsmasq."""
    table = iptables_manager.ipv4['filter']
    for port in [67, 53]:
        for proto in ['udp', 'tcp']:
            args = {'dev': dev, 'port': port, 'proto': proto}
            table.add_rule('INPUT',
                           '-i %(dev)s -p %(proto)s -m %(proto)s '
                           '--dport %(port)s -j ACCEPT' % args)
    iptables_manager.apply()


def _remove_dnsmasq_accept_rules(dev):
    """Remove DHCP and DNS traffic allowed through to dnsmasq."""
    table = iptables_manager.ipv4['filter']
    for port in [67, 53]:
        for proto in ['udp', 'tcp']:
            args = {'dev': dev, 'port': port, 'proto': proto}
            table.remove_rule('INPUT',
                           '-i %(dev)s -p %(proto)s -m %(proto)s '
                           '--dport %(port)s -j ACCEPT' % args)
    iptables_manager.apply()


def _add_dhcp_mangle_rule(dev):
    if not os.path.exists('/dev/vhost-net'):
        return
    table = iptables_manager.ipv4['mangle']
    table.add_rule('POSTROUTING',
                   '-o %s -p udp -m udp --dport 68 -j CHECKSUM '
                   '--checksum-fill' % dev)
    iptables_manager.apply()


def _remove_dhcp_mangle_rule(dev):
    table = iptables_manager.ipv4['mangle']
    table.remove_rule('POSTROUTING',
                      '-o %s -p udp -m udp --dport 68 -j CHECKSUM '
                      '--checksum-fill' % dev)
    iptables_manager.apply()


def get_dhcp_opts(context, network_ref):
    """Get network's hosts config in dhcp-opts format."""
    hosts = []
    host = None
    if network_ref['multi_host']:
        host = CONF.host
    data = db.network_get_associated_fixed_ips(context,
                                               network_ref['id'],
                                               host=host)

    if data:
        instance_set = set([datum['instance_uuid'] for datum in data])
        default_gw_vif = {}
        for instance_uuid in instance_set:
            vifs = db.virtual_interface_get_by_instance(context,
                                                        instance_uuid)
            if vifs:
                #offer a default gateway to the first virtual interface
                default_gw_vif[instance_uuid] = vifs[0]['id']

        for datum in data:
            instance_uuid = datum['instance_uuid']
            if instance_uuid in default_gw_vif:
                # we don't want default gateway for this fixed ip
                if default_gw_vif[instance_uuid] != datum['vif_id']:
                    hosts.append(_host_dhcp_opts(datum))
    return '\n'.join(hosts)


def release_dhcp(dev, address, mac_address):
    utils.execute('dhcp_release', dev, address, mac_address, run_as_root=True)


def update_dhcp(context, dev, network_ref):
    conffile = _dhcp_file(dev, 'conf')
    write_to_file(conffile, get_dhcp_hosts(context, network_ref))
    restart_dhcp(context, dev, network_ref)


def update_dns(context, dev, network_ref):
    hostsfile = _dhcp_file(dev, 'hosts')
    write_to_file(hostsfile, get_dns_hosts(context, network_ref))
    restart_dhcp(context, dev, network_ref)


def update_dhcp_hostfile_with_text(dev, hosts_text):
    conffile = _dhcp_file(dev, 'conf')
    write_to_file(conffile, hosts_text)


def kill_dhcp(dev):
    pid = _dnsmasq_pid_for(dev)
    if pid:
        # Check that the process exists and looks like a dnsmasq process
        conffile = _dhcp_file(dev, 'conf')
        out, _err = _execute('cat', '/proc/%d/cmdline' % pid,
                             check_exit_code=False)
        if conffile.split('/')[-1] in out:
            _execute('kill', '-9', pid, run_as_root=True)
        else:
            LOG.debug(_('Pid %d is stale, skip killing dnsmasq'), pid)
    _remove_dnsmasq_accept_rules(dev)
    _remove_dhcp_mangle_rule(dev)


# NOTE(ja): Sending a HUP only reloads the hostfile, so any
#           configuration options (like dchp-range, vlan, ...)
#           aren't reloaded.
@lockutils.synchronized('dnsmasq_start', 'nova-')
def restart_dhcp(context, dev, network_ref):
    """(Re)starts a dnsmasq server for a given network.

    If a dnsmasq instance is already running then send a HUP
    signal causing it to reload, otherwise spawn a new instance.

    """
    conffile = _dhcp_file(dev, 'conf')

    if CONF.use_single_default_gateway:
        # NOTE(vish): this will have serious performance implications if we
        #             are not in multi_host mode.
        optsfile = _dhcp_file(dev, 'opts')
        write_to_file(optsfile, get_dhcp_opts(context, network_ref))
        os.chmod(optsfile, 0644)

    if network_ref['multi_host']:
        _add_dhcp_mangle_rule(dev)

    # Make sure dnsmasq can actually read it (it setuid()s to "nobody")
    os.chmod(conffile, 0644)

    pid = _dnsmasq_pid_for(dev)

    # if dnsmasq is already running, then tell it to reload
    if pid:
        out, _err = _execute('cat', '/proc/%d/cmdline' % pid,
                             check_exit_code=False)
        # Using symlinks can cause problems here so just compare the name
        # of the file itself
        if conffile.split('/')[-1] in out:
            try:
                _execute('kill', '-HUP', pid, run_as_root=True)
                _add_dnsmasq_accept_rules(dev)
                return
            except Exception as exc:  # pylint: disable=W0703
                LOG.error(_('Hupping dnsmasq threw %s'), exc)
        else:
            LOG.debug(_('Pid %d is stale, relaunching dnsmasq'), pid)

    cmd = ['env',
           'CONFIG_FILE=%s' % jsonutils.dumps(CONF.dhcpbridge_flagfile),
           'NETWORK_ID=%s' % str(network_ref['id']),
           'dnsmasq',
           '--strict-order',
           '--bind-interfaces',
           '--conf-file=%s' % CONF.dnsmasq_config_file,
           '--pid-file=%s' % _dhcp_file(dev, 'pid'),
           '--listen-address=%s' % network_ref['dhcp_server'],
           '--except-interface=lo',
           '--dhcp-range=set:%s,%s,static,%s,%ss' %
                         (network_ref['label'],
                          network_ref['dhcp_start'],
                          network_ref['netmask'],
                          CONF.dhcp_lease_time),
           '--dhcp-lease-max=%s' % len(netaddr.IPNetwork(network_ref['cidr'])),
           '--dhcp-hostsfile=%s' % _dhcp_file(dev, 'conf'),
           '--dhcp-script=%s' % CONF.dhcpbridge,
           '--leasefile-ro']

    # dnsmasq currently gives an error for an empty domain,
    # rather than ignoring.  So only specify it if defined.
    if CONF.dhcp_domain:
        cmd.append('--domain=%s' % CONF.dhcp_domain)

    dns_servers = set(CONF.dns_server)
    if CONF.use_network_dns_servers:
        if network_ref.get('dns1'):
            dns_servers.add(network_ref.get('dns1'))
        if network_ref.get('dns2'):
            dns_servers.add(network_ref.get('dns2'))
    if network_ref['multi_host'] or dns_servers:
        cmd.append('--no-hosts')
    if network_ref['multi_host']:
        cmd.append('--addn-hosts=%s' % _dhcp_file(dev, 'hosts'))
    if dns_servers:
        cmd.append('--no-resolv')
    for dns_server in dns_servers:
        cmd.append('--server=%s' % dns_server)
    if CONF.use_single_default_gateway:
        cmd += ['--dhcp-optsfile=%s' % _dhcp_file(dev, 'opts')]

    _execute(*cmd, run_as_root=True)

    _add_dnsmasq_accept_rules(dev)


@lockutils.synchronized('radvd_start', 'nova-')
def update_ra(context, dev, network_ref):
    conffile = _ra_file(dev, 'conf')
    conf_str = """
interface %s
{
   AdvSendAdvert on;
   MinRtrAdvInterval 3;
   MaxRtrAdvInterval 10;
   prefix %s
   {
        AdvOnLink on;
        AdvAutonomous on;
   };
};
""" % (dev, network_ref['cidr_v6'])
    write_to_file(conffile, conf_str)

    # Make sure radvd can actually read it (it setuid()s to "nobody")
    os.chmod(conffile, 0644)

    pid = _ra_pid_for(dev)

    # if radvd is already running, then tell it to reload
    if pid:
        out, _err = _execute('cat', '/proc/%d/cmdline'
                             % pid, check_exit_code=False)
        if conffile in out:
            try:
                _execute('kill', pid, run_as_root=True)
            except Exception as exc:  # pylint: disable=W0703
                LOG.error(_('killing radvd threw %s'), exc)
        else:
            LOG.debug(_('Pid %d is stale, relaunching radvd'), pid)

    cmd = ['radvd',
           '-C', '%s' % _ra_file(dev, 'conf'),
           '-p', '%s' % _ra_file(dev, 'pid')]

    _execute(*cmd, run_as_root=True)


def _host_lease(data):
    """Return a host string for an address in leasefile format."""
    timestamp = timeutils.utcnow()
    seconds_since_epoch = calendar.timegm(timestamp.utctimetuple())
    return '%d %s %s %s *' % (seconds_since_epoch + CONF.dhcp_lease_time,
                              data['vif_address'],
                              data['address'],
                              data['instance_hostname'] or '*')


def _host_dhcp_network(data):
    return 'NW-%s' % data['vif_id']


def _host_dhcp(data):
    """Return a host string for an address in dhcp-host format."""
    if CONF.use_single_default_gateway:
        return '%s,%s.%s,%s,%s' % (data['vif_address'],
                               data['instance_hostname'],
                               CONF.dhcp_domain,
                               data['address'],
                               'net:' + _host_dhcp_network(data))
    else:
        return '%s,%s.%s,%s' % (data['vif_address'],
                               data['instance_hostname'],
                               CONF.dhcp_domain,
                               data['address'])


def _host_dns(data):
    return '%s\t%s.%s' % (data['address'],
                          data['instance_hostname'],
                          CONF.dhcp_domain)


def _host_dhcp_opts(data):
    """Return an empty gateway option."""
    return '%s,%s' % (_host_dhcp_network(data), 3)


def _execute(*cmd, **kwargs):
    """Wrapper around utils._execute for fake_network."""
    if CONF.fake_network:
        LOG.debug('FAKE NET: %s', ' '.join(map(str, cmd)))
        return 'fake', 0
    else:
        return utils.execute(*cmd, **kwargs)


def device_exists(device):
    """Check if ethernet device exists."""
    (_out, err) = _execute('ip', 'link', 'show', 'dev', device,
                           check_exit_code=False, run_as_root=True)
    return not err


def _dhcp_file(dev, kind):
    """Return path to a pid, leases, hosts or conf file for a bridge/device."""
    fileutils.ensure_tree(CONF.networks_path)
    return os.path.abspath('%s/nova-%s.%s' % (CONF.networks_path,
                                              dev,
                                              kind))


def _ra_file(dev, kind):
    """Return path to a pid or conf file for a bridge/device."""
    fileutils.ensure_tree(CONF.networks_path)
    return os.path.abspath('%s/nova-ra-%s.%s' % (CONF.networks_path,
                                              dev,
                                              kind))


def _dnsmasq_pid_for(dev):
    """Returns the pid for prior dnsmasq instance for a bridge/device.

    Returns None if no pid file exists.

    If machine has rebooted pid might be incorrect (caller should check).

    """
    pid_file = _dhcp_file(dev, 'pid')

    if os.path.exists(pid_file):
        try:
            with open(pid_file, 'r') as f:
                return int(f.read())
        except (ValueError, IOError):
            return None


def _ra_pid_for(dev):
    """Returns the pid for prior radvd instance for a bridge/device.

    Returns None if no pid file exists.

    If machine has rebooted pid might be incorrect (caller should check).

    """
    pid_file = _ra_file(dev, 'pid')

    if os.path.exists(pid_file):
        with open(pid_file, 'r') as f:
            return int(f.read())


def _ip_bridge_cmd(action, params, device):
    """Build commands to add/del ips to bridges/devices."""
    cmd = ['ip', 'addr', action]
    cmd.extend(params)
    cmd.extend(['dev', device])
    return cmd


def _create_veth_pair(dev1_name, dev2_name):
    """Create a pair of veth devices with the specified names,
    deleting any previous devices with those names.
    """
    for dev in [dev1_name, dev2_name]:
        delete_net_dev(dev)

    utils.execute('ip', 'link', 'add', dev1_name, 'type', 'veth', 'peer',
                  'name', dev2_name, run_as_root=True)
    for dev in [dev1_name, dev2_name]:
        utils.execute('ip', 'link', 'set', dev, 'up', run_as_root=True)
        utils.execute('ip', 'link', 'set', dev, 'promisc', 'on',
                      run_as_root=True)


def create_ovs_vif_port(bridge, dev, iface_id, mac, instance_id):
    utils.execute('ovs-vsctl', '--', '--may-exist', 'add-port',
                  bridge, dev,
                  '--', 'set', 'Interface', dev,
                  'external-ids:iface-id=%s' % iface_id,
                  'external-ids:iface-status=active',
                  'external-ids:attached-mac=%s' % mac,
                  'external-ids:vm-uuid=%s' % instance_id,
                  run_as_root=True)


def delete_ovs_vif_port(bridge, dev):
    utils.execute('ovs-vsctl', 'del-port', bridge, dev,
                  run_as_root=True)
    delete_net_dev(dev)


def create_tap_dev(dev, mac_address=None):
    if not device_exists(dev):
        try:
            # First, try with 'ip'
            utils.execute('ip', 'tuntap', 'add', dev, 'mode', 'tap',
                          run_as_root=True, check_exit_code=[0, 2, 254])
        except exception.ProcessExecutionError:
            # Second option: tunctl
            utils.execute('tunctl', '-b', '-t', dev, run_as_root=True)
        if mac_address:
            utils.execute('ip', 'link', 'set', dev, 'address', mac_address,
                          run_as_root=True, check_exit_code=[0, 2, 254])
        utils.execute('ip', 'link', 'set', dev, 'up', run_as_root=True,
                      check_exit_code=[0, 2, 254])


def delete_net_dev(dev):
    """Delete a network device only if it exists."""
    if device_exists(dev):
        try:
            utils.execute('ip', 'link', 'delete', dev, run_as_root=True,
                          check_exit_code=[0, 2, 254])
            LOG.debug(_("Net device removed: '%s'"), dev)
        except exception.ProcessExecutionError:
            with excutils.save_and_reraise_exception():
                LOG.error(_("Failed removing net device: '%s'"), dev)


# Similar to compute virt layers, the Linux network node
# code uses a flexible driver model to support different ways
# of creating ethernet interfaces and attaching them to the network.
# In the case of a network host, these interfaces
# act as gateway/dhcp/vpn/etc. endpoints not VM interfaces.
interface_driver = None


def _get_interface_driver():
    global interface_driver
    if not interface_driver:
        interface_driver = importutils.import_object(
                CONF.linuxnet_interface_driver)
    return interface_driver


def plug(network, mac_address, gateway=True):
    return _get_interface_driver().plug(network, mac_address, gateway)


def unplug(network):
    return _get_interface_driver().unplug(network)


def get_dev(network):
    return _get_interface_driver().get_dev(network)


class LinuxNetInterfaceDriver(object):
    """
    Abstract class that defines generic network host API
    for for all Linux interface drivers.
    """

    def plug(self, network, mac_address):
        """Create Linux device, return device name."""
        raise NotImplementedError()

    def unplug(self, network):
        """Destroy Linux device, return device name."""
        raise NotImplementedError()

    def get_dev(self, network):
        """Get device name."""
        raise NotImplementedError()


# plugs interfaces using Linux Bridge
class LinuxBridgeInterfaceDriver(LinuxNetInterfaceDriver):

    def plug(self, network, mac_address, gateway=True):
        vlan = network.get('vlan')
        if vlan is not None:
            iface = CONF.vlan_interface or network['bridge_interface']
            LinuxBridgeInterfaceDriver.ensure_vlan_bridge(
                           vlan,
                           network['bridge'],
                           iface,
                           network,
                           mac_address)
            iface = 'vlan%s' % vlan
        else:
            iface = CONF.flat_interface or network['bridge_interface']
            LinuxBridgeInterfaceDriver.ensure_bridge(
                          network['bridge'],
                          iface,
                          network, gateway)

        if CONF.share_dhcp_address:
            isolate_dhcp_address(iface, network['dhcp_server'])
        # NOTE(vish): applying here so we don't get a lock conflict
        iptables_manager.apply()
        return network['bridge']

    def unplug(self, network, gateway=True):
        vlan = network.get('vlan')
        if vlan is not None:
            iface = 'vlan%s' % vlan
            LinuxBridgeInterfaceDriver.remove_vlan_bridge(vlan,
                                                          network['bridge'])
        else:
            iface = CONF.flat_interface or network['bridge_interface']
            LinuxBridgeInterfaceDriver.remove_bridge(network['bridge'],
                                                     gateway)

        if CONF.share_dhcp_address:
            remove_isolate_dhcp_address(iface, network['dhcp_server'])

        iptables_manager.apply()
        return self.get_dev(network)

    def get_dev(self, network):
        return network['bridge']

    @classmethod
    def ensure_vlan_bridge(_self, vlan_num, bridge, bridge_interface,
                                            net_attrs=None, mac_address=None):
        """Create a vlan and bridge unless they already exist."""
        interface = LinuxBridgeInterfaceDriver.ensure_vlan(vlan_num,
                                               bridge_interface, mac_address)
        LinuxBridgeInterfaceDriver.ensure_bridge(bridge, interface, net_attrs)
        return interface

    @classmethod
    def remove_vlan_bridge(cls, vlan_num, bridge):
        """Delete a bridge and vlan."""
        LinuxBridgeInterfaceDriver.remove_bridge(bridge)
        LinuxBridgeInterfaceDriver.remove_vlan(vlan_num)

    @classmethod
    @lockutils.synchronized('lock_vlan', 'nova-', external=True)
    def ensure_vlan(_self, vlan_num, bridge_interface, mac_address=None):
        """Create a vlan unless it already exists."""
        interface = 'vlan%s' % vlan_num
        if not device_exists(interface):
            LOG.debug(_('Starting VLAN interface %s'), interface)
            _execute('ip', 'link', 'add', 'link', bridge_interface,
                     'name', interface, 'type', 'vlan',
                     'id', vlan_num, run_as_root=True,
                     check_exit_code=[0, 2, 254])
            # (danwent) the bridge will inherit this address, so we want to
            # make sure it is the value set from the NetworkManager
            if mac_address:
                _execute('ip', 'link', 'set', interface, 'address',
                         mac_address, run_as_root=True,
                         check_exit_code=[0, 2, 254])
            _execute('ip', 'link', 'set', interface, 'up', run_as_root=True,
                     check_exit_code=[0, 2, 254])
            if CONF.network_device_mtu:
                _execute('ip', 'link', 'set', interface, 'mtu',
                         CONF.network_device_mtu, run_as_root=True,
                         check_exit_code=[0, 2, 254])
        return interface

    @classmethod
    @lockutils.synchronized('lock_vlan', 'nova-', external=True)
    def remove_vlan(cls, vlan_num):
        """Delete a vlan."""
        vlan_interface = 'vlan%s' % vlan_num
        delete_net_dev(vlan_interface)

    @classmethod
    @lockutils.synchronized('lock_bridge', 'nova-', external=True)
    def ensure_bridge(_self, bridge, interface, net_attrs=None, gateway=True,
                      filtering=True):
        """Create a bridge unless it already exists.

        :param interface: the interface to create the bridge on.
        :param net_attrs: dictionary with  attributes used to create bridge.
        :param gateway: whether or not the bridge is a gateway.
        :param filtering: whether or not to create filters on the bridge.

        If net_attrs is set, it will add the net_attrs['gateway'] to the bridge
        using net_attrs['broadcast'] and net_attrs['cidr'].  It will also add
        the ip_v6 address specified in net_attrs['cidr_v6'] if use_ipv6 is set.

        The code will attempt to move any ips that already exist on the
        interface onto the bridge and reset the default gateway if necessary.

        """
        if not device_exists(bridge):
            LOG.debug(_('Starting Bridge %s'), bridge)
            _execute('brctl', 'addbr', bridge, run_as_root=True)
            _execute('brctl', 'setfd', bridge, 0, run_as_root=True)
            # _execute('brctl setageing %s 10' % bridge, run_as_root=True)
            _execute('brctl', 'stp', bridge, 'off', run_as_root=True)
            # (danwent) bridge device MAC address can't be set directly.
            # instead it inherits the MAC address of the first device on the
            # bridge, which will either be the vlan interface, or a
            # physical NIC.
            _execute('ip', 'link', 'set', bridge, 'up', run_as_root=True)

        if interface:
            msg = _('Adding interface %(interface)s to bridge %(bridge)s')
            LOG.debug(msg % locals())
            out, err = _execute('brctl', 'addif', bridge, interface,
                                check_exit_code=False, run_as_root=True)

            # NOTE(vish): This will break if there is already an ip on the
            #             interface, so we move any ips to the bridge
            # NOTE(danms): We also need to copy routes to the bridge so as
            #              not to break existing connectivity on the interface
            old_routes = []
            out, err = _execute('ip', 'route', 'show', 'dev', interface)
            for line in out.split('\n'):
                fields = line.split()
                if fields and 'via' in fields:
                    old_routes.append(fields)
                    _execute('ip', 'route', 'del', *fields,
                             run_as_root=True)
            out, err = _execute('ip', 'addr', 'show', 'dev', interface,
                                'scope', 'global', run_as_root=True)
            for line in out.split('\n'):
                fields = line.split()
                if fields and fields[0] == 'inet':
                    if fields[-2] == 'secondary':
                        params = fields[1:-2]
                    else:
                        params = fields[1:-1]
                    _execute(*_ip_bridge_cmd('del', params, fields[-1]),
                             run_as_root=True, check_exit_code=[0, 2, 254])
                    _execute(*_ip_bridge_cmd('add', params, bridge),
                             run_as_root=True, check_exit_code=[0, 2, 254])
            for fields in old_routes:
                _execute('ip', 'route', 'add', *fields,
                         run_as_root=True)

            if (err and err != "device %s is already a member of a bridge;"
                     "can't enslave it to bridge %s.\n" % (interface, bridge)):
                msg = _('Failed to add interface: %s') % err
                raise exception.NovaException(msg)

        if filtering:
            # Don't forward traffic unless we were told to be a gateway
            ipv4_filter = iptables_manager.ipv4['filter']
            if gateway:
                for rule in get_gateway_rules(bridge):
                    ipv4_filter.add_rule(*rule)
            else:
                ipv4_filter.add_rule('FORWARD',
                                     '--in-interface %s -j DROP' % bridge)
                ipv4_filter.add_rule('FORWARD',
                                     '--out-interface %s -j DROP' % bridge)

    @classmethod
    @lockutils.synchronized('lock_bridge', 'nova-', external=True)
    def remove_bridge(cls, bridge, gateway=True, filtering=True):
        """Delete a bridge."""
        if not device_exists(bridge):
            return
        else:
            if filtering:
                ipv4_filter = iptables_manager.ipv4['filter']
                if gateway:
                    for rule in get_gateway_rules(bridge):
                        ipv4_filter.remove_rule(*rule)
                else:
                    ipv4_filter.remove_rule('FORWARD',
                                    '--in-interface %s -j DROP' % bridge)
                    ipv4_filter.remove_rule('FORWARD',
                                    '--out-interface %s -j DROP' % bridge)
            delete_net_dev(bridge)


@lockutils.synchronized('ebtables', 'nova-', external=True)
def ensure_ebtables_rules(rules, table='filter'):
    for rule in rules:
        cmd = ['ebtables', '-t', table, '-D'] + rule.split()
        _execute(*cmd, check_exit_code=False, run_as_root=True)
        cmd[3] = '-I'
        _execute(*cmd, run_as_root=True)


@lockutils.synchronized('ebtables', 'nova-', external=True)
def remove_ebtables_rules(rules, table='filter'):
    for rule in rules:
        cmd = ['ebtables', '-t', table, '-D'] + rule.split()
        _execute(*cmd, check_exit_code=False, run_as_root=True)


def isolate_dhcp_address(interface, address):
    # block arp traffic to address across the interface
    rules = []
    rules.append('INPUT -p ARP -i %s --arp-ip-dst %s -j DROP'
                 % (interface, address))
    rules.append('OUTPUT -p ARP -o %s --arp-ip-src %s -j DROP'
                 % (interface, address))
    # NOTE(vish): the above is not possible with iptables/arptables
    ensure_ebtables_rules(rules)
    # block dhcp broadcast traffic across the interface
    ipv4_filter = iptables_manager.ipv4['filter']
    ipv4_filter.add_rule('FORWARD',
                         '-m physdev --physdev-in %s -d 255.255.255.255 '
                         '-p udp --dport 67 -j DROP' % interface, top=True)
    ipv4_filter.add_rule('FORWARD',
                         '-m physdev --physdev-out %s -d 255.255.255.255 '
                         '-p udp --dport 67 -j DROP' % interface, top=True)
    # block ip traffic to address across the interface
    ipv4_filter.add_rule('FORWARD',
                         '-m physdev --physdev-in %s -d %s -j DROP'
                         % (interface, address), top=True)
    ipv4_filter.add_rule('FORWARD',
                         '-m physdev --physdev-out %s -s %s -j DROP'
                         % (interface, address), top=True)


def remove_isolate_dhcp_address(interface, address):
    # block arp traffic to address across the interface
    rules = []
    rules.append('INPUT -p ARP -i %s --arp-ip-dst %s -j DROP'
                 % (interface, address))
    rules.append('OUTPUT -p ARP -o %s --arp-ip-src %s -j DROP'
                 % (interface, address))
    remove_ebtables_rules(rules)
    # NOTE(vish): the above is not possible with iptables/arptables
    # block dhcp broadcast traffic across the interface
    ipv4_filter = iptables_manager.ipv4['filter']
    ipv4_filter.remove_rule('FORWARD',
                         '-m physdev --physdev-in %s -d 255.255.255.255 '
                         '-p udp --dport 67 -j DROP' % interface, top=True)
    ipv4_filter.remove_rule('FORWARD',
                         '-m physdev --physdev-out %s -d 255.255.255.255 '
                         '-p udp --dport 67 -j DROP' % interface, top=True)
    # block ip traffic to address across the interface
    ipv4_filter.remove_rule('FORWARD',
                         '-m physdev --physdev-in %s -d %s -j DROP'
                         % (interface, address), top=True)
    ipv4_filter.remove_rule('FORWARD',
                         '-m physdev --physdev-out %s -s %s -j DROP'
                         % (interface, address), top=True)


def get_gateway_rules(bridge):
    interfaces = CONF.forward_bridge_interface
    if 'all' in interfaces:
        return [('FORWARD', '-i %s -j ACCEPT' % bridge),
                ('FORWARD', '-o %s -j ACCEPT' % bridge)]
    rules = []
    for iface in CONF.forward_bridge_interface:
        if iface:
            rules.append(('FORWARD', '-i %s -o %s -j ACCEPT' % (bridge,
                                                                iface)))
            rules.append(('FORWARD', '-i %s -o %s -j ACCEPT' % (iface,
                                                                bridge)))
    rules.append(('FORWARD', '-i %s -o %s -j ACCEPT' % (bridge, bridge)))
    rules.append(('FORWARD', '-i %s -j DROP' % bridge))
    rules.append(('FORWARD', '-o %s -j DROP' % bridge))
    return rules


# plugs interfaces using Open vSwitch
class LinuxOVSInterfaceDriver(LinuxNetInterfaceDriver):

    def plug(self, network, mac_address, gateway=True):
        dev = self.get_dev(network)
        if not device_exists(dev):
            bridge = CONF.linuxnet_ovs_integration_bridge
            _execute('ovs-vsctl',
                     '--', '--may-exist', 'add-port', bridge, dev,
                     '--', 'set', 'Interface', dev, 'type=internal',
                     '--', 'set', 'Interface', dev,
                     'external-ids:iface-id=%s' % dev,
                     '--', 'set', 'Interface', dev,
                     'external-ids:iface-status=active',
                     '--', 'set', 'Interface', dev,
                     'external-ids:attached-mac=%s' % mac_address,
                     run_as_root=True)
            _execute('ip', 'link', 'set', dev, 'address', mac_address,
                     run_as_root=True)
            if CONF.network_device_mtu:
                _execute('ip', 'link', 'set', dev, 'mtu',
                         CONF.network_device_mtu, run_as_root=True)
            _execute('ip', 'link', 'set', dev, 'up', run_as_root=True)
            if not gateway:
                # If we weren't instructed to act as a gateway then add the
                # appropriate flows to block all non-dhcp traffic.
                _execute('ovs-ofctl',
                         'add-flow', bridge, 'priority=1,actions=drop',
                         run_as_root=True)
                _execute('ovs-ofctl', 'add-flow', bridge,
                         'udp,tp_dst=67,dl_dst=%s,priority=2,actions=normal' %
                         mac_address, run_as_root=True)
                # .. and make sure iptbles won't forward it as well.
                iptables_manager.ipv4['filter'].add_rule('FORWARD',
                        '--in-interface %s -j DROP' % bridge)
                iptables_manager.ipv4['filter'].add_rule('FORWARD',
                        '--out-interface %s -j DROP' % bridge)
            else:
                for rule in get_gateway_rules(bridge):
                    iptables_manager.ipv4['filter'].add_rule(*rule)

        return dev

    def unplug(self, network):
        dev = self.get_dev(network)
        bridge = CONF.linuxnet_ovs_integration_bridge
        _execute('ovs-vsctl', '--', '--if-exists', 'del-port',
                 bridge, dev, run_as_root=True)
        return dev

    def get_dev(self, network):
        dev = 'gw-' + str(network['uuid'][0:11])
        return dev


# plugs interfaces using Linux Bridge when using QuantumManager
class QuantumLinuxBridgeInterfaceDriver(LinuxNetInterfaceDriver):

    BRIDGE_NAME_PREFIX = 'brq'
    GATEWAY_INTERFACE_PREFIX = 'gw-'

    def plug(self, network, mac_address, gateway=True):
        dev = self.get_dev(network)
        bridge = self.get_bridge(network)
        if not gateway:
            # If we weren't instructed to act as a gateway then add the
            # appropriate flows to block all non-dhcp traffic.
            # .. and make sure iptbles won't forward it as well.
            iptables_manager.ipv4['filter'].add_rule('FORWARD',
                    '--in-interface %s -j DROP' % bridge)
            iptables_manager.ipv4['filter'].add_rule('FORWARD',
                    '--out-interface %s -j DROP' % bridge)
            return bridge
        else:
            for rule in get_gateway_rules(bridge):
                iptables_manager.ipv4['filter'].add_rule(*rule)

        create_tap_dev(dev, mac_address)

        if not device_exists(bridge):
            LOG.debug(_("Starting bridge %s "), bridge)
            utils.execute('brctl', 'addbr', bridge, run_as_root=True)
            utils.execute('brctl', 'setfd', bridge, str(0), run_as_root=True)
            utils.execute('brctl', 'stp', bridge, 'off', run_as_root=True)
            utils.execute('ip', 'link', 'set', bridge, 'address', mac_address,
                          run_as_root=True, check_exit_code=[0, 2, 254])
            utils.execute('ip', 'link', 'set', bridge, 'up', run_as_root=True,
                          check_exit_code=[0, 2, 254])
            LOG.debug(_("Done starting bridge %s"), bridge)

            full_ip = '%s/%s' % (network['dhcp_server'],
                                 network['cidr'].rpartition('/')[2])
            utils.execute('ip', 'address', 'add', full_ip, 'dev', bridge,
                          run_as_root=True, check_exit_code=[0, 2, 254])

        return dev

    def unplug(self, network):
        dev = self.get_dev(network)
        if not device_exists(dev):
            return None
        else:
            delete_net_dev(dev)
            return dev

    def get_dev(self, network):
        dev = self.GATEWAY_INTERFACE_PREFIX + str(network['uuid'][0:11])
        return dev

    def get_bridge(self, network):
        bridge = self.BRIDGE_NAME_PREFIX + str(network['uuid'][0:11])
        return bridge

iptables_manager = IptablesManager()
