// Copyright 2013 Canonical Ltd.
// Licensed under the AGPLv3, see LICENCE file for details.

package network

import (
	"bytes"
	"net"
)

// Private network ranges for IPv4.
// See: http://tools.ietf.org/html/rfc1918
//
// TODO(dimitern) 2014-06-10 LP bug #1328429
// Improve private IP address detection here
// by adding missing CIDRs for private networks.
var (
	classAPrivate = mustParseCIDR("10.0.0.0/8")
	classBPrivate = mustParseCIDR("172.16.0.0/12")
	classCPrivate = mustParseCIDR("192.168.0.0/16")
)

func mustParseCIDR(s string) *net.IPNet {
	_, net, err := net.ParseCIDR(s)
	if err != nil {
		panic(err)
	}
	return net
}

// AddressType represents the possible ways of specifying a machine location by
// either a hostname resolvable by dns lookup, or IPv4 or IPv6 address.
type AddressType string

const (
	HostName    AddressType = "hostname"
	IPv4Address AddressType = "ipv4"
	IPv6Address AddressType = "ipv6"
)

// Scope denotes the context a location may apply to. If a name or
// address can be reached from the wider internet, it is considered
// public. A private network address is either specific to the cloud
// or cloud subnet a machine belongs to, or to the machine itself for
// containers.
type Scope string

const (
	ScopeUnknown      Scope = ""
	ScopePublic       Scope = "public"
	ScopeCloudLocal   Scope = "local-cloud"
	ScopeMachineLocal Scope = "local-machine"
)

// Address represents the location of a machine, including metadata
// about what kind of location the address describes.
type Address struct {
	Value       string
	Type        AddressType
	NetworkName string
	Scope
}

// String returns a string representation of the address,
// in the form: scope:address(network name);
// for example:
//
//	public:c2-54-226-162-124.compute-1.amazonaws.com(ec2network)
//
// If the scope is NetworkUnknown, the initial scope: prefix will
// be omitted. If the NetworkName is blank, the (network name) suffix
// will be omitted.
func (a Address) String() string {
	var buf bytes.Buffer
	if a.Scope != ScopeUnknown {
		buf.WriteString(string(a.Scope))
		buf.WriteByte(':')
	}
	buf.WriteString(a.Value)
	if a.NetworkName != "" {
		buf.WriteByte('(')
		buf.WriteString(a.NetworkName)
		buf.WriteByte(')')
	}
	return buf.String()
}

// NewAddresses is a convenience function to create addresses from a string slice
func NewAddresses(inAddresses ...string) (outAddresses []Address) {
	for _, address := range inAddresses {
		outAddresses = append(outAddresses, NewAddress(address, ScopeUnknown))
	}
	return outAddresses
}

// DeriveAddressType attempts to detect the type of address given.
func DeriveAddressType(value string) AddressType {
	ip := net.ParseIP(value)
	switch {
	case ip == nil:
		// TODO(gz): Check value is a valid hostname
		return HostName
	case ip.To4() != nil:
		return IPv4Address
	case ip.To16() != nil:
		return IPv6Address
	default:
		panic("Unknown form of IP address")
	}
}

func isIPv4PrivateNetworkAddress(ip net.IP) bool {
	return classAPrivate.Contains(ip) ||
		classBPrivate.Contains(ip) ||
		classCPrivate.Contains(ip)
}

// deriveScope attempts to derive the network scope from an address's
// type and value, returning the original network scope if no
// deduction can be made.
func deriveScope(addr Address) Scope {
	if addr.Type == HostName {
		return addr.Scope
	}
	ip := net.ParseIP(addr.Value)
	if ip == nil {
		return addr.Scope
	}
	if ip.IsLoopback() {
		return ScopeMachineLocal
	}
	switch addr.Type {
	case IPv4Address:
		if isIPv4PrivateNetworkAddress(ip) {
			return ScopeCloudLocal
		}
		// If it's not loopback, and it's not a private
		// network address, then it's publicly routable.
		return ScopePublic
	case IPv6Address:
		// TODO(axw) check for IPv6 unique local address, if/when we care.
	}
	return addr.Scope
}

// NewAddress creates a new Address, deriving its type from the value.
//
// If the specified scope is ScopeUnknown, then NewAddress will
// attempt derive the scope based on reserved IP address ranges.
func NewAddress(value string, scope Scope) Address {
	addr := Address{
		Value: value,
		Type:  DeriveAddressType(value),
		Scope: scope,
	}
	if scope == ScopeUnknown {
		addr.Scope = deriveScope(addr)
	}
	return addr
}

// SelectPublicAddress picks one address from a slice that would
// be appropriate to display as a publicly accessible endpoint.
// If there are no suitable addresses, the empty string is returned.
//
// TODO(dimitern) Make this work properly with #IPv6 addresses.
func SelectPublicAddress(addresses []Address) string {
	index := bestAddressIndex(len(addresses), func(i int) Address {
		return addresses[i]
	}, publicMatch)
	if index < 0 {
		return ""
	}
	return addresses[index].Value
}

func SelectPublicHostPort(hps []HostPort) string {
	index := bestAddressIndex(len(hps), func(i int) Address {
		return hps[i].Address
	}, publicMatch)
	if index < 0 {
		return ""
	}
	return hps[index].NetAddr()
}

// SelectInternalAddress picks one address from a slice that can be
// used as an endpoint for juju internal communication.
// If there are no suitable addresses, the empty string is returned.
func SelectInternalAddress(addresses []Address, machineLocal bool) string {
	index := bestAddressIndex(len(addresses), func(i int) Address {
		return addresses[i]
	}, internalAddressMatcher(machineLocal))
	if index < 0 {
		return ""
	}
	return addresses[index].Value
}

// SelectInternalHostPort picks one HostPort from a slice that can be
// used as an endpoint for juju internal communication and returns it
// in its NetAddr form.
// If there are no suitable addresses, the empty string is returned.
func SelectInternalHostPort(hps []HostPort, machineLocal bool) string {
	index := bestAddressIndex(len(hps), func(i int) Address {
		return hps[i].Address
	}, internalAddressMatcher(machineLocal))
	if index < 0 {
		return ""
	}
	return hps[index].NetAddr()
}

func publicMatch(addr Address) scopeMatch {
	switch addr.Scope {
	case ScopePublic:
		return exactScope
	case ScopeCloudLocal, ScopeUnknown:
		return fallbackScope
	}
	return invalidScope
}

func internalAddressMatcher(machineLocal bool) func(Address) scopeMatch {
	if machineLocal {
		return cloudOrMachineLocalMatch
	}
	return cloudLocalMatch
}

func cloudLocalMatch(addr Address) scopeMatch {
	switch addr.Scope {
	case ScopeCloudLocal:
		return exactScope
	case ScopePublic, ScopeUnknown:
		return fallbackScope
	}
	return invalidScope
}

func cloudOrMachineLocalMatch(addr Address) scopeMatch {
	if addr.Scope == ScopeMachineLocal {
		return exactScope
	}
	return cloudLocalMatch(addr)
}

type scopeMatch int

const (
	invalidScope scopeMatch = iota
	exactScope
	fallbackScope
)

// bestAddressIndex returns the index of the first address
// with an exactly matching scope, or the first address with
// a matching fallback scope if there are no exact matches.
// If there are no suitable addresses, -1 is returned.
func bestAddressIndex(numAddr int, getAddr func(i int) Address, match func(addr Address) scopeMatch) int {
	fallbackAddressIndex := -1
	for i := 0; i < numAddr; i++ {
		addr := getAddr(i)
		if addr.Type != IPv6Address {
			switch match(addr) {
			case exactScope:
				return i
			case fallbackScope:
				// Use the first fallback address if there are no exact matches.
				if fallbackAddressIndex == -1 {
					fallbackAddressIndex = i
				}
			}
		}
	}
	return fallbackAddressIndex
}
