/*-
 * The code in this file was written by Eivind Eklund <perhaps@yes.no>,
 * who places it in the public domain without restriction.
 *
 * $FreeBSD: stable/2.2/usr.sbin/ppp/alias_cmd.c 50952 1999-09-05 11:54:50Z peter $
 */

#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <sys/un.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>

#ifdef __OpenBSD__
#include "alias.h"
#else
#include <alias.h>
#endif
#include "defs.h"
#include "command.h"
#include "log.h"
#include "alias_cmd.h"
#include "descriptor.h"
#include "prompt.h"
#include "timer.h"
#include "fsm.h"
#include "slcompress.h"
#include "throughput.h"
#include "iplist.h"
#include "mbuf.h"
#include "lqr.h"
#include "hdlc.h"
#include "ipcp.h"
#include "lcp.h"
#include "ccp.h"
#include "link.h"
#include "mp.h"
#include "filter.h"
#include "bundle.h"


static int StrToAddr(const char *, struct in_addr *);
static int StrToPort(const char *, u_short *, const char *);
static int StrToAddrAndPort(const char *, struct in_addr *, u_short *, const char *);


int
alias_RedirectPort(struct cmdargs const *arg)
{
  if (!arg->bundle->AliasEnabled) {
    prompt_Printf(arg->prompt, "Alias not enabled\n");
    return 1;
  } else if (arg->argc == arg->argn+3) {
    char proto_constant;
    const char *proto;
    u_short local_port;
    u_short alias_port;
    int error;
    struct in_addr local_addr;
    struct in_addr null_addr;
    struct alias_link *link;

    proto = arg->argv[arg->argn];
    if (strcmp(proto, "tcp") == 0) {
      proto_constant = IPPROTO_TCP;
    } else if (strcmp(proto, "udp") == 0) {
      proto_constant = IPPROTO_UDP;
    } else {
      prompt_Printf(arg->prompt, "port redirect: protocol must be"
                    " tcp or udp\n");
      prompt_Printf(arg->prompt, "Usage: alias %s %s\n", arg->cmd->name,
		    arg->cmd->syntax);
      return 1;
    }

    error = StrToAddrAndPort(arg->argv[arg->argn+1], &local_addr, &local_port,
                             proto);
    if (error) {
      prompt_Printf(arg->prompt, "port redirect: error reading"
                    " local addr:port\n");
      prompt_Printf(arg->prompt, "Usage: alias %s %s\n", arg->cmd->name,
                    arg->cmd->syntax);
      return 1;
    }
    error = StrToPort(arg->argv[arg->argn+2], &alias_port, proto);
    if (error) {
      prompt_Printf(arg->prompt, "port redirect: error reading alias port\n");
      prompt_Printf(arg->prompt, "Usage: alias %s %s\n", arg->cmd->name,
                    arg->cmd->syntax);
      return 1;
    }
    null_addr.s_addr = INADDR_ANY;

    link = PacketAliasRedirectPort(local_addr, local_port,
				   null_addr, 0,
				   null_addr, alias_port,
				   proto_constant);

    if (link == NULL)
      prompt_Printf(arg->prompt, "port redirect: error returned by packed"
	      " aliasing engine (code=%d)\n", error);
  } else
    return -1;

  return 0;
}


int
alias_RedirectAddr(struct cmdargs const *arg)
{
  if (!arg->bundle->AliasEnabled) {
    prompt_Printf(arg->prompt, "alias not enabled\n");
    return 1;
  } else if (arg->argc == arg->argn+2) {
    int error;
    struct in_addr local_addr;
    struct in_addr alias_addr;
    struct alias_link *link;

    error = StrToAddr(arg->argv[arg->argn], &local_addr);
    if (error) {
      prompt_Printf(arg->prompt, "address redirect: invalid local address\n");
      return 1;
    }
    error = StrToAddr(arg->argv[arg->argn+1], &alias_addr);
    if (error) {
      prompt_Printf(arg->prompt, "address redirect: invalid alias address\n");
      prompt_Printf(arg->prompt, "Usage: alias %s %s\n", arg->cmd->name,
                    arg->cmd->syntax);
      return 1;
    }
    link = PacketAliasRedirectAddr(local_addr, alias_addr);
    if (link == NULL) {
      prompt_Printf(arg->prompt, "address redirect: packet aliasing"
                    " engine error\n");
      prompt_Printf(arg->prompt, "Usage: alias %s %s\n", arg->cmd->name,
                    arg->cmd->syntax);
    }
  } else
    return -1;

  return 0;
}


static int
StrToAddr(const char *str, struct in_addr *addr)
{
  struct hostent *hp;

  if (inet_aton(str, addr))
    return 0;

  hp = gethostbyname(str);
  if (!hp) {
    log_Printf(LogWARN, "StrToAddr: Unknown host %s.\n", str);
    return -1;
  }
  *addr = *((struct in_addr *) hp->h_addr);
  return 0;
}


static int
StrToPort(const char *str, u_short *port, const char *proto)
{
  int iport;
  struct servent *sp;
  char *end;

  iport = strtol(str, &end, 10);
  if (end != str) {
    *port = htons(iport);
    return 0;
  }
  sp = getservbyname(str, proto);
  if (!sp) {
    log_Printf(LogWARN, "StrToAddr: Unknown port or service %s/%s.\n",
	      str, proto);
    return -1;
  }
  *port = sp->s_port;
  return 0;
}


static int
StrToAddrAndPort(const char *str, struct in_addr *addr, u_short *port, const char *proto)
{
  char *colon;
  int res;

  colon = strchr(str, ':');
  if (!colon) {
    log_Printf(LogWARN, "StrToAddrAndPort: %s is missing port number.\n", str);
    return -1;
  }

  *colon = '\0';		/* Cheat the const-ness ! */
  res = StrToAddr(str, addr);
  *colon = ':';			/* Cheat the const-ness ! */
  if (res != 0)
    return -1;

  return StrToPort(colon+1, port, proto);
}
