<?php
/**
 * XMP Reader - Read XMP data
 *
 * Zoph is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Zoph is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with Zoph; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @author Jeroen Roos
 * @package Zoph
 */

namespace xmp;

use file;
use XMLReader;

/**
 * @author Jeroen Roos
 * @package Zoph
 */
class reader  {

    private static $containers = array("rdf:Seq", "rdf:Bag", "rdf:Alt");
    private $xmpdata = array();
    private $XMPs = array();

    /**
     * Create new xmp/reader object
     * @param string file to read
     */
    public function __construct(file $file) {
        $this->file = $file;
    }

    public function getXMP() {
        $this->file->getMime();
        switch ($this->file->type) {
            case "image":
                $this->getXMPfromJPEG();
                break;
            case "xml":
                $this->getXMPfromXML();
                break;
            default:
                throw new \Exception("Unknown filetype " . $this->file->type);
        }
        $this->readXMPs();

        $return = array();
        foreach ($this->xmpdata as $xmp) {
            $return[] = new data($xmp);
        }

        return $return;
    }

    /**
     * Read XMP-xml from an XML-file
     */
    private function getXMPfromXML() {
        $this->XMPs=array(file_get_contents($this->file));
    }

    private function readXMPs() {
        foreach ($this->XMPs as $xmp) {
            $this->xmpdata[]=$this->readXMP($xmp);
        }
    }

    /**
     * Decode XMP data from XML
     * @param string XML
     * @return array XMPdata
     */
    public function readXMP($xmp) {
        $xml = new XMLReader();
        $xml->XML($xmp);
        return $this->decodeXML($xml);
    }

    /**
     * Read XMP-xml from a JPEG
     * Read JPG in chunks of 1024 bytes to keep memory usage low.
     */
    private function getXMPfromJPEG() {
        $fp = fopen($this->file, "rb");
        $buffer = "";
        $open = false;

        while (($data = fread($fp, 1024)) !== false) {
            if($data === "") {
                break;
            }
            $buffer .= $data;
            if(!$open) {
                $openpos =strpos($buffer, "<x:xmpmeta");
                if($openpos !== false) {
                    $open = true;
                } else {
                    // keep 12 bytes of buffer, in case the XMP is right on the edge
                    $buffer = substr($buffer, strlen($buffer) - 12);
                }
            } else {
                $closepos =strpos($buffer, "</x:xmpmeta>");
                if($closepos !== false) {
                    $this->XMPs[] = substr($buffer, $openpos, ($closepos - $openpos) + 12);
                    $open = false;
                    $buffer = substr($buffer, $closepos + 11);
                }
            }
        }
    }

    private function decodeXMLcontainer(XMLReader $xml) {
        $node=array();
        if(in_array($xml->name, self::$containers)) {
            $container = $xml->name;
        }
        while(!($xml->nodeType==XMLReader::END_ELEMENT && $xml->name==$container)) {
            $xml->read();
            if ($xml->nodeType==XMLReader::ELEMENT && ($xml->name=="rdf:li" || in_array($xml->name, self::$containers))) {
                $node[]=$this->decodeXMLcontainer($xml);
            } else if ($xml->nodeType==XMLReader::TEXT) {
                return $xml->value;
            } else if ($xml->nodeType==XMLReader::END_ELEMENT) {
            } else if ($xml->nodeType==XMLReader::SIGNIFICANT_WHITESPACE) {
            } else {
                /** @todo define exception and provide proper error */
                throw new \Exception("Failed");
            }
        }
        return $node;
    }

    private function decodeXML(XMLReader $xml) {
        $node=array();
        while($xml->read()) {
            if ($xml->nodeType==XMLReader::ELEMENT) {
                $nodename = $xml->name;

                $empty=$xml->isEmptyElement;

                if ($xml->hasAttributes) {
                    $attr=array();
                    $xml->moveToFirstAttribute();

                    do {
                        $attr[$xml->name] = $xml->value;
                    } while ($xml->moveToNextAttribute());
                    if (!$empty) {
                        $node[$nodename]=array_merge($attr, $this->decodeXML($xml));
                    } else {
                        $node[$nodename]=$attr;
                    }
                } else if ($xml->hasValue) {
                    $node[] = $xml->value;
                } else {
                    if (in_array($nodename, self::$containers)) {
                        $node[$nodename]=$this->decodeXMLcontainer($xml);
                    } else {
                        $node[$nodename]=$this->decodeXML($xml);
                    }
                }

            } else if ($xml->nodeType==XMLReader::TEXT) {
                $node[]=$xml->value;
            } else if ($xml->nodeType==XMLReader::END_ELEMENT) {
                return $node;
            } else if ($xml->nodeType==XMLReader::WHITESPACE) {
            } else if ($xml->nodeType==XMLReader::SIGNIFICANT_WHITESPACE) {
            } else {
                $node = $xml->value;
            }
        }
        return $node;
    }
}
?>
