/*
 * Copyright (c) 2021, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test Loading CDS archived heap objects into SerialGC
 * @bug 8234679
 * @requires vm.cds
 * @requires vm.gc.Serial
 * @requires vm.gc.G1
 *
 * @comment don't run this test if any -XX::+Use???GC options are specified, since they will
 *          interfere with the test.
 * @requires vm.gc == null
 *
 * @library /test/lib /test/hotspot/jtreg/runtime/cds/appcds
 * @compile test-classes/Hello.java
 * @run driver TestSerialGCWithCDS
 */

// Below is exactly the same as above, except:
// - requires vm.bits == "64"
// - extra argument "false"

/*
 * @test Loading CDS archived heap objects into SerialGC
 * @bug 8234679
 * @requires vm.cds
 * @requires vm.gc.Serial
 * @requires vm.gc.G1
 * @requires vm.bits == "64"
 *
 * @comment don't run this test if any -XX::+Use???GC options are specified, since they will
 *          interfere with the test.
 * @requires vm.gc == null
 *
 * @library /test/lib /test/hotspot/jtreg/runtime/cds/appcds
 * @compile test-classes/Hello.java
 * @run driver TestSerialGCWithCDS false
 */

import jdk.test.lib.Platform;
import jdk.test.lib.process.OutputAnalyzer;

public class TestSerialGCWithCDS {
    public final static String HELLO = "Hello World";
    static String helloJar;
    static boolean useCompressedOops = true;

    public static void main(String... args) throws Exception {
        helloJar = JarBuilder.build("hello", "Hello");

        if (args.length > 0 && args[0].equals("false")) {
            useCompressedOops = false;
        }

        // Check if we can use SerialGC during dump time, or run time, or both.
        test(false, true);
        test(true,  false);
        test(true,  true);

        // We usually have 2 heap regions. To increase test coverage, we can have 3 heap regions
        // by using "-Xmx256m -XX:ObjectAlignmentInBytes=64"
        if (Platform.is64bit()) {
            test(false, true, /*useSmallRegions=*/true);
        }
    }

    final static String G1 = "-XX:+UseG1GC";
    final static String Serial = "-XX:+UseSerialGC";

    static void test(boolean dumpWithSerial, boolean execWithSerial) throws Exception {
        test(dumpWithSerial, execWithSerial, false);
    }

    static void test(boolean dumpWithSerial, boolean execWithSerial, boolean useSmallRegions) throws Exception {
        String DUMMY = "-showversion"; // A harmless option that doesn't doesn't do anything except for printing out the version
        String dumpGC = dumpWithSerial ? Serial : G1;
        String execGC = execWithSerial ? Serial : G1;
        String small1 = useSmallRegions ? "-Xmx256m" : DUMMY;
        String small2 = useSmallRegions ? "-XX:ObjectAlignmentInBytes=64" : DUMMY;
        String errMsg = "Cannot use CDS heap data. Selected GC not compatible -XX:-UseCompressedOops";
        String coops;
        if (Platform.is64bit()) {
            coops = useCompressedOops ? "-XX:+UseCompressedOops" : "-XX:-UseCompressedOops";
        } else {
            coops = DUMMY;
        }
        OutputAnalyzer out;

        System.out.println("0. Dump with " + dumpGC);
        out = TestCommon.dump(helloJar,
                              new String[] {"Hello"},
                              dumpGC,
                              small1,
                              small2,
                              coops,
                              "-Xlog:cds");
        out.shouldContain("Dumping shared data to file:");
        out.shouldHaveExitValue(0);

        System.out.println("1. Exec with " + execGC);
        out = TestCommon.exec(helloJar,
                              execGC,
                              small1,
                              small2,
                              coops,
                              "-Xlog:cds",
                              "Hello");
        out.shouldNotContain(errMsg);

        System.out.println("2. Exec with " + execGC + " and test ArchiveRelocationMode=0");
        out = TestCommon.exec(helloJar,
                              execGC,
                              small1,
                              small2,
                              coops,
                              "-Xlog:cds,cds+heap",
                              "-XX:ArchiveRelocationMode=0", // may relocate shared metadata
                              "Hello");
        out.shouldNotContain(errMsg);

        int n = 2;
        if (execWithSerial == true) {
            // At exec time, try to load archived objects into a small SerialGC heap that may be too small.
            String[] sizes = {
                "4m",   // usually this will success load the archived heap
                "2m",   // usually this will fail to load the archived heap, but app can launch
                        // or fail with "GC triggered before VM initialization completed"
                "1m"    // usually this will cause VM launch to fail with "Too small maximum heap"
            };
            for (String sz : sizes) {
                String xmx = "-Xmx" + sz;
                System.out.println("=======\n" + n + ". Exec with " + execGC + " " + xmx);
                out = TestCommon.exec(helloJar,
                                      execGC,
                                      small1,
                                      small2,
                                      xmx,
                                      coops,
                                      "-Xlog:cds",
                                      "Hello");
                if (out.getExitValue() == 0) {
                    out.shouldNotContain(errMsg);
                } else {
                    out.shouldNotHaveFatalError();
                }
                n++;
            }
        }
    }
}
