#!/usr/bin/perl -w

#######################################################################
#
# pod2wiki - A utility to convert Pod documents to Wiki format.
#
# Copyright John McNamara, jmcnamara@cpan.org
#
# Documentation after __END__
#

use strict;
use Pod::Simple::Wiki;
use Getopt::Long;
use Pod::Usage;

my $man      = 0;
my $help     = 0;
my $style    = 'wiki';
my $whine    = 1;
my $errata   = 1;
my $complain = 0;
my $encoding = '';

GetOptions(
            'help|?'     => \$help,
            'man'        => \$man,
            'style=s'    => \$style,
            'whine!'     => \$whine,
            'errata!'    => \$errata,
            'complain!'  => \$complain,
            'encoding=s' => \$encoding,
          ) or pod2usage(2);

pod2usage(1) if $help;
pod2usage(-verbose => 2) if $man;


# From the Pod::Usage pod:
pod2usage() if @ARGV == 0 && -t STDIN;


my $parser = Pod::Simple::Wiki->new($style);
   $parser->no_whining(not $whine);
   $parser->no_errata_section(not $errata);
   $parser->complain_stderr($complain);

if (defined $ARGV[0]) {
    open IN, $ARGV[0]  or die "Couldn't open $ARGV[0]: $!\n";
} else {
    *IN = *STDIN;
}

if (defined $ARGV[1]) {
    open OUT, ">$ARGV[1]" or die "Couldn't open $ARGV[1]: $!\n";
    $parser->output_fh(*OUT);
}

# Encode the output filehandle if required.
if ($] >= 5.008 && $encoding) {
    binmode $parser->{output_fh}, $encoding;
}

$parser->parse_file(*IN);


__END__

=head1 NAME

pod2wiki - A utility to convert Pod documents to Wiki format.

=head1 SYNOPSIS

pod2wiki [--style --noerrata --help --man] podfile [outfile]

    Options:
        --style      wiki style (defaults to wiki. See --help)
        --noerrata   don't generate a "POD ERRORS" section
        --help       brief help message
        --man        full documentation


=head1 DESCRIPTION

This program is used for converting Pod text to Wiki text.

Pod is Perl's I<Plain Old Documentation> format. See C<man perlpod> or C<perldoc perlpod>.

A Wiki is a user extensible web site. It uses very simple mark-up that is converted to Html. For an introduction to Wikis see: L<http://en.wikipedia.org/wiki/Wiki>


=head1 OPTIONS

=over 4

=item B<podfile>

The input file that contains the Pod file to be converted. It can also be stdin.

=item B<outfile>

The converted output file in wiki format. Defaults to stdout if not specified.

=item B<--style or -s>

Sets the wiki style of the output. If no C<style> is specified the program defaults to C<wiki>. The available options are:

=over 4

=item wiki

This is the original Wiki format as used on Ward Cunningham's Portland repository of Patterns. See L<http://c2.com/cgi/wiki>

=item kwiki

This is the format as used by Brian Ingerson's Kwiki: L<http://www.kwiki.org>

=item usemod

This is the format used by the Usemod wikis. See: L<http://www.usemod.com/cgi-bin/wiki.pl>

=item twiki

This is the format used by TWiki wikis. See: L<http://twiki.org/>

=item tiddlywiki

This is the format used by the TiddlyWiki. See: L<http://www.tiddlywiki.com/>

=item textile

The Textile markup format as used on GitHub. See: L<http://textile.thresholdstate.com/>

=item wikipedia or mediawiki

This is the format used by Wikipedia and MediaWiki wikis. See: L<http://www.mediawiki.org/>

=item markdown

This is the format used by GitHub and other sites. See: L<http://daringfireball.net/projects/markdown/syntax>.


=item moinmoin

This is the format used by MoinMoin wikis. See: L<http://moinmo.in/MoinMoinWiki>

=item confluence

This is the format used by Confluence. See: L<http://www.atlassian.com/software/confluence/>

=back


=item B<--encoding>

Specify the encoding for the output filehandle:

    --encoding=":utf8"

Refer to C<binmode> in L<perlfunc> for more details. This option is only available in Perl 5.8 and later.

=item B<--noerrata or -noe>

Don't generate a "POD ERRORS" section at the end of the document. Equivalent to the C<Pod::Simple::no_errata_section()> method.


=item B<--help or -h>

Print a brief help message and exits.


=item B<--man or -m>

Prints the manual page and exits.

=back


=head1 DISCLAIMER OF WARRANTY

Please refer to the DISCLAIMER OF WARRANTY in L<Pod::Simple::Wiki>.


=head1 AUTHOR

John McNamara jmcnamara@cpan.org


=head1 COPYRIGHT

(c) MMIII-MMXV John McNamara.

All Rights Reserved. This module is free software. It may be used, redistributed and/or modified under the same terms as Perl itself.

=cut
